// -*- Mode: Go; indent-tabs-mode: t -*-

/*
 * Copyright (C) 2016 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package kmod_test

import (
	"path/filepath"
	"testing"

	. "gopkg.in/check.v1"

	"github.com/snapcore/snapd/dirs"
	"github.com/snapcore/snapd/interfaces"
	"github.com/snapcore/snapd/interfaces/ifacetest"
	"github.com/snapcore/snapd/interfaces/kmod"
	"github.com/snapcore/snapd/osutil"
	"github.com/snapcore/snapd/snap"
	"github.com/snapcore/snapd/testutil"
	"github.com/snapcore/snapd/timings"
)

func Test(t *testing.T) {
	TestingT(t)
}

type backendSuite struct {
	ifacetest.BackendSuite
	modprobeCmd *testutil.MockCmd
	meas        *timings.Span
}

var _ = Suite(&backendSuite{})

var testedConfinementOpts = []interfaces.ConfinementOptions{
	{},
	{DevMode: true},
	{JailMode: true},
	{Classic: true},
}

func (s *backendSuite) SetUpTest(c *C) {
	s.Backend = &kmod.Backend{}
	s.BackendSuite.SetUpTest(c)
	c.Assert(s.Repo.AddBackend(s.Backend), IsNil)
	s.modprobeCmd = testutil.MockCommand(c, "modprobe", "")

	perf := timings.New(nil)
	s.meas = perf.StartSpan("", "")
}

func (s *backendSuite) TearDownTest(c *C) {
	s.modprobeCmd.Restore()
	s.BackendSuite.TearDownTest(c)
}

func (s *backendSuite) TestName(c *C) {
	c.Check(s.Backend.Name(), Equals, interfaces.SecurityKMod)
}

func (s *backendSuite) TestInstallingSnapCreatesModulesConf(c *C) {
	// NOTE: Hand out a permanent snippet so that .conf file is generated.
	s.Iface.KModPermanentSlotCallback = func(spec *kmod.Specification, slot *snap.SlotInfo) error {
		spec.AddModule("module1")
		spec.AddModule("module2")
		return nil
	}

	path := filepath.Join(dirs.SnapKModModulesDir, "snap.samba.conf")
	c.Assert(osutil.FileExists(path), Equals, false)

	for _, opts := range testedConfinementOpts {
		s.modprobeCmd.ForgetCalls()
		snapInfo := s.InstallSnap(c, opts, "", ifacetest.SambaYamlV1, 0)

		c.Assert(osutil.FileExists(path), Equals, true)
		c.Assert(path, testutil.FileEquals, "# This file is automatically generated.\nmodule1\nmodule2\n")

		c.Assert(s.modprobeCmd.Calls(), DeepEquals, [][]string{
			{"modprobe", "--syslog", "module1"},
			{"modprobe", "--syslog", "module2"},
		})
		s.RemoveSnap(c, snapInfo)
	}
}

func (s *backendSuite) TestRemovingSnapRemovesModulesConf(c *C) {
	// NOTE: Hand out a permanent snippet so that .conf file is generated.
	s.Iface.KModPermanentSlotCallback = func(spec *kmod.Specification, slot *snap.SlotInfo) error {
		spec.AddModule("module1")
		spec.AddModule("module2")
		return nil
	}

	path := filepath.Join(dirs.SnapKModModulesDir, "snap.samba.conf")
	c.Assert(osutil.FileExists(path), Equals, false)

	for _, opts := range testedConfinementOpts {
		snapInfo := s.InstallSnap(c, opts, "", ifacetest.SambaYamlV1, 0)
		c.Assert(osutil.FileExists(path), Equals, true)
		s.RemoveSnap(c, snapInfo)
		c.Assert(osutil.FileExists(path), Equals, false)
	}
}

func (s *backendSuite) TestInstallingSnapCreatesModprobeConf(c *C) {
	s.Iface.KModPermanentSlotCallback = func(spec *kmod.Specification, slot *snap.SlotInfo) error {
		spec.AddModule("module1")
		spec.SetModuleOptions("module1", "opt1=true opt2=2")
		spec.DisallowModule("module2")
		return nil
	}

	modulesPath := filepath.Join(dirs.SnapKModModulesDir, "snap.samba.conf")
	c.Assert(osutil.FileExists(modulesPath), Equals, false)
	modprobePath := filepath.Join(dirs.SnapKModModprobeDir, "snap.samba.conf")
	c.Assert(osutil.FileExists(modprobePath), Equals, false)

	for _, opts := range testedConfinementOpts {
		s.modprobeCmd.ForgetCalls()
		snapInfo := s.InstallSnap(c, opts, "", ifacetest.SambaYamlV1, 0)

		c.Assert(osutil.FileExists(modulesPath), Equals, true)
		c.Assert(modulesPath, testutil.FileEquals, "# This file is automatically generated.\nmodule1\n")

		c.Assert(osutil.FileExists(modprobePath), Equals, true)
		c.Assert(modprobePath, testutil.FileEquals, `# Generated by snapd. Do not edit

blacklist module2
options module1 opt1=true opt2=2
`)

		c.Assert(s.modprobeCmd.Calls(), DeepEquals, [][]string{
			{"modprobe", "--syslog", "module1"},
		})
		s.RemoveSnap(c, snapInfo)
	}
}

func (s *backendSuite) TestRemovingSnapRemovesModprobeConf(c *C) {
	s.Iface.KModPermanentSlotCallback = func(spec *kmod.Specification, slot *snap.SlotInfo) error {
		spec.AddModule("module1")
		spec.SetModuleOptions("module1", "opt1=true opt2=2")
		spec.DisallowModule("module2")
		return nil
	}

	modulesPath := filepath.Join(dirs.SnapKModModulesDir, "snap.samba.conf")
	c.Assert(osutil.FileExists(modulesPath), Equals, false)
	modprobePath := filepath.Join(dirs.SnapKModModprobeDir, "snap.samba.conf")
	c.Assert(osutil.FileExists(modprobePath), Equals, false)

	for _, opts := range testedConfinementOpts {
		snapInfo := s.InstallSnap(c, opts, "", ifacetest.SambaYamlV1, 0)
		c.Assert(osutil.FileExists(modulesPath), Equals, true)
		c.Assert(osutil.FileExists(modprobePath), Equals, true)
		s.RemoveSnap(c, snapInfo)
		c.Assert(osutil.FileExists(modulesPath), Equals, false)
		c.Assert(osutil.FileExists(modprobePath), Equals, false)
	}
}

func (s *backendSuite) TestSecurityIsStable(c *C) {
	// NOTE: Hand out a permanent snippet so that .conf file is generated.
	s.Iface.KModPermanentSlotCallback = func(spec *kmod.Specification, slot *snap.SlotInfo) error {
		spec.AddModule("module1")
		spec.AddModule("module2")
		return nil
	}

	for _, opts := range testedConfinementOpts {
		snapInfo := s.InstallSnap(c, opts, "", ifacetest.SambaYamlV1, 0)
		appSet, err := interfaces.NewSnapAppSet(snapInfo, nil)
		c.Assert(err, IsNil)
		s.modprobeCmd.ForgetCalls()
		err = s.Backend.Setup(appSet, opts, s.Repo, s.meas)
		c.Assert(err, IsNil)
		// modules conf is not re-loaded when nothing changes
		c.Check(s.modprobeCmd.Calls(), HasLen, 0)
		s.RemoveSnap(c, snapInfo)
	}
}

func (s *backendSuite) TestSandboxFeatures(c *C) {
	c.Assert(s.Backend.SandboxFeatures(), DeepEquals, []string{"mediated-modprobe"})
}
