################################################################################
### Part of the surveillance package, http://surveillance.r-forge.r-project.org
### Free software under the terms of the GNU General Public License, version 2,
### a copy of which is available at http://www.r-project.org/Licenses/.
###
### Helper functions for neighbourhood weight matrices in hhh4()
###
### Copyright (C) 2012-2013 Sebastian Meyer
### $Revision: 641 $
### $Date: 2013-09-05 15:31:34 +0200 (Don, 05 Sep 2013) $
################################################################################


### check ne$weights specification

checkNeighbourhood <- function (neighbourhood)
{
    ## setValidity() in sts.R only guarantees correct 'dim' and 'dimnames'
    ## we also assert numeric or logical matrix with non-NA entries
    ## FIXME: However, we currently don't check for symmetry and for zeros on
    ## the diagonal...
    stopifnot(is.matrix(neighbourhood),
              nrow(neighbourhood) == ncol(neighbourhood),
              is.numeric(neighbourhood) | is.logical(neighbourhood),
              is.finite(neighbourhood))
    invisible(TRUE)
}

checkWeightsArray <- function (W, nUnits, nTime, name)
{
    if (!is.array(W))
        stop("'", name, "' must return a matrix/array")
    if (any(dim(W)[1:2] != nUnits) || isTRUE(dim(W)[3] != nTime))
        stop("'", name, "' must conform to dimensions ",
             nUnits, " x ", nUnits, " (x ", nTime, ")")
    if (any(is.na(W)))
        stop("missing values in '", name, "' are not allowed")
    diags <- if (is.matrix(W)) diag(W) else apply(W, 3, diag)
    if (any(diags != 0))
        stop("'", name, "' must only contain zeroes on the diagonal",
             if (!is.matrix(W)) "s")
}

checkWeights <- function (weights, nUnits, nTime,
                          nbmat, data)  # only used for parametric weights
{
    name <- "control$ne$weights"

    ## check specification
    testweights <- if (is.array(weights)) weights else {
        if (is.list(weights) && checkWeightsFUN(weights)
            && checkNeighbourhood(nbmat)) {
            if (all(nbmat %in% 0:1))
                warning("'", deparse(substitute(nbmat)),
                        "' is binary (should contain",
                        " general neighbourhood orders)")
            weights$w(weights$initial, nbmat, data)
        } else {
            stop("'", name, "' must be a matrix/array or a list of functions")
        }
    }
    
    ## apply matrix/array checks
    if (is.list(weights)) { # parametric weights
        checkWeightsArray(testweights, nUnits, nTime, name=paste0(name, "$w"))
        dim.d <- length(weights$initial)
        dw <- weights$dw(weights$initial, nbmat, data)
        d2w <- weights$d2w(weights$initial, nbmat, data)
        if (dim.d == 1L) {
            checkWeightsArray(dw, nUnits, nTime, name=paste0(name, "$dw"))
            checkWeightsArray(d2w, nUnits, nTime, name=paste0(name, "$d2w"))
        } else {
            if (!is.list(dw) || length(dw) != dim.d)
                stop("'", name, "$dw' must return a list (of matrices/arrays)",
                     " of length ", dim.d)
            if (!is.list(d2w) || length(d2w) != dim.d*(dim.d+1)/2)
                stop("'", name, "$d2w' must return a list (of matrices/arrays)",
                     " of length ", dim.d*(dim.d+1)/2)
            lapply(dw, checkWeightsArray, nUnits, nTime,
                   name=paste0(name, "$dw[[i]]"))
            lapply(d2w, checkWeightsArray, nUnits, nTime,
                   name=paste0(name, "$d2w[[i]]"))
        }
    } else checkWeightsArray(testweights, nUnits, nTime, name=name)
    
    ## Done
    invisible(TRUE)
}



### calculate the weighted sum of counts of adjacent (or all other) regions
### i.e. the nTime x nUnit matrix with elements ne_ti = sum_j w_jit * y_jt
## W is either a nUnits x nUnits matrix of time-constant weights w_ji
## or a nUnits x nUnits x nTime array of time-varying weights

weightedSumNE <- function (observed, weights, lag)
{
  dimY <- dim(observed)
  nTime <- dimY[1L]
  nUnits <- dimY[2L]
  tY <- t(observed)                     # -> nUnits x nTime
  
  res <- apply(weights, 2L, function (wi)
               ## if dim(weights)==2 (time-constant weights), length(wi)=nUnits,
               ## if dim(weights)==3, wi is a matrix of size nUnits x nTime
               .colSums(tY * wi, nUnits, nTime, na.rm=TRUE))
  
  rbind(matrix(NA_real_, lag, nUnits),
        res[seq_len(nTime-lag),,drop=FALSE])
}

## slower alternative, where the weights are always converted to a 3D array
weightedSumNE.old <- function(observed, weights, lag)
{
  dimY <- dim(observed)
  nTime <- dimY[1L]
  nUnits <- dimY[2L]
  
  nhood <- array(weights, c(nUnits,nUnits,nTime))
  
  res <- matrix(NA_real_, nrow = nTime, ncol = nUnits,
                dimnames = list(NULL, colnames(observed)))
  for(i in seq_len(nUnits)){
    weights.i <- t(nhood[,i,])
    weightedObs <- observed * weights.i
    res[,i] <- rowSums(weightedObs, na.rm=TRUE)
  }
  
  rbind(matrix(NA_real_, lag, nUnits), head(res, nTime-lag))
}



###############################################
### predefined parametric weight structures ###
###############################################


### check parametric weights specification consisting of a list of:
## - three functions: w, dw, and d2w
## - a vector of initial parameter values

checkWeightsFUN <- function (object)
{
    fnames <- paste0(c("","d","d2"), "w")
    if (any(!sapply(object[fnames], is.function)))
        stop("parametric weights require functions ",
             paste0("'", fnames, "'", collapse=", "))
    if (any(!sapply(object[fnames], function(FUN) length(formals(FUN)) >= 3L)))
        stop("parametric weights functions must accept (not necessarily use)",
             "\n  at least 3 arguments (parameter vector, ",
             "neighbourhood order matrix, data)")
    if (!is.vector(object$initial, mode="numeric") ||
        length(object$initial) == 0L)
        stop("parametric weights require initial parameter values")
    TRUE
}


### Construct weight matrix wji according to the Zeta-distribution with respect
### to the orders of neighbourhood (in nbmat, as e.g. obtained from nbOrder()),
### optionally fulfilling rowSums(wji) = 1
## As a formula (for j != i, otherwise wji = 0):
## - for shared=TRUE: wji = pzeta(oji; d, maxlag) / sum_k I(ojk == oji)
## - for shared=FALSE: wji = pzeta(oji; d, maxlag) / sum_k pzeta(ojk; d, maxlag)
## Here, oji = oij is the order of nb of i and j,
## and pzeta(o; d, m) = o^-d / sum_{r=1}^m r^-d is the Zeta-distribution
## on 1:m (also called Zipf's law).
## For shared=TRUE (not recommended) and normalize=FALSE, maxlag should be
## <= min_j(max_i(oji)), such that every region has neighbours up to order 
## 'maxlag' and higher orders can not be infected. Otherwise, regions with not
## as high-order neighbours would not sum their weights to 1 (but lower).
## For shared=FALSE, maxlag=max(nbmat) yields the weights
## wji = oji^-d / sum_k ojk^-d
## In both cases, maxlag=1 yields the classical weights wji=1/nj.

zetaweights <- function (nbmat, d = 1, maxlag = max(nbmat), normalize = FALSE)
{
    ## raw (non-normalized) zeta-distribution on 1:maxlag
    zeta <- c(0, seq_len(maxlag)^-d)  # first 0 is for lag 0 (i.e., diag(nbmat))

    ## replace order by zetaweight of that order
    wji <- zeta[nbmat + 1L]           # results in vector
    wji[is.na(wji)] <- 0              # for lags > maxlag

    ## set dim and names
    dim(wji) <- dimW <- dim(nbmat)
    dimnames(wji) <- dimnames(nbmat)

    ## if (shared) {
    ##     ## multiplicity of orders by row: dim(nbmat)==dim(multbyrow)
    ##     multbyrow <- t(apply(nbmat, 1, function(x) table(x)[as.character(x)]))
    ##     ## neighbours of same order share the zetaweight for that order
    ##     wji <- wji / sum(zeta) / multbyrow
    ## }
    if (normalize) { # normalize such that each row sums to 1
        wji / .rowSums(wji, dimW[1], dimW[2])
    } else wji
}



### powerlaw weights
## in the non-truncated case, i.e. maxlag = max(nbmat),
## the raw powerlaw weights are defined as w_ji = o_ji^-d,
## and with (row-)normalization we have    w_ji = o_ji^-d / sum_k o_jk^-d

powerlaw <- function (maxlag, normalize = TRUE, log = FALSE,
                      initial = if (log) 0 else 1)
{
    if (missing(maxlag)) {
        stop("'maxlag' must be specified (e.g. maximum neighbourhood order)")
        ## specifying 'maxlag' in zetaweights is actually optional since it has
        ## the default value max(nbmat). however, repeatedly asking for this
        ## maximum would be really inefficient.
    } else stopifnot(isScalar(maxlag))

    ## main function which returns the weight matrix
    weights.call <- call("zetaweights",
                         quote(nbmat), quote(d), maxlag, normalize)
    weights <- as.function(c(alist(d=, nbmat=, ...=), call("{", weights.call)),
                           envir=.GlobalEnv)
    if (log) { # the parameter d is interpreted on log-scale
        ## we prepend the necessary conversion d <- exp(d)
        body(weights) <- as.call(append(as.list(body(weights)),
                                        quote(d <- exp(d)), after=1))
    }
    
    ## construct derivatives with respect to "d" (or log(d), respectively)
    dweights <- d2weights <- as.function(c(alist(d=, nbmat=, ...=), quote({})),
                                         envir=.GlobalEnv)
    header <- c(
        if (log) quote(d <- exp(d)),    # such that d is again on original scale
        substitute(W <- weights.call, list(weights.call=weights.call)),
        expression(
            nUnits <- nrow(W),           # such that we can use .rowSums()
            is.na(nbmat) <- nbmat == 0L, # w_jj(d)=0 => w_jj'(d)=0, sum over j!=i,
            logo <- log(nbmat)           # have to set NA because we do log(o)
            )
        )
    footer <- expression(deriv[is.na(deriv)] <- 0, deriv)

    ## first derivative
    tmp1 <- expression(
        norm <- .rowSums(nbmat^-d, nUnits, nUnits, na.rm=TRUE),
        tmpnorm <- .rowSums(nbmat^-d * -log(nbmat), nUnits, nUnits, na.rm=TRUE) / norm,
        tmp1 <- logo + tmpnorm
        )
    deriv1 <- if (normalize) {
        expression(deriv <- W * -tmp1)
    } else expression(deriv <- W * -logo)
    body(dweights) <- as.call(c(as.name("{"),
            header,
            if (normalize) tmp1,
            deriv1,
            if (log) expression(deriv <- deriv * d), # this is the non-log d
            footer
        ))

    ## second derivative
    body(d2weights) <- as.call(c(as.name("{"),
            header,
            if (normalize) {
                c(tmp1, expression(
                    tmp2 <- .rowSums(nbmat^-d * log(nbmat)^2, nUnits, nUnits,
                                     na.rm=TRUE) / norm - tmpnorm^2,
                    deriv <- W * (tmp1^2 - tmp2)
                    ))
            } else expression(deriv <- W * logo^2),
            if (log) c(
                do.call("substitute", list(deriv1[[1]], list(deriv=as.name("deriv1")))),
                expression(deriv <- deriv * d^2 + deriv1 * d) # this is the non-log d
                ),
            footer
        ))

    ## return list of functions
    list(w=weights, dw=dweights, d2w=d2weights, initial=initial)
}


