/*
 *  Copyright 2011 Wolfgang Koller - http://www.gofg.at/
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

#include <QGeoPositionInfo>
#include <QGeoCoordinate>

#include "geolocation.h"

Geolocation::Geolocation(Cordova *cordova) : CPlugin(cordova) {
    m_geoPositionInfoSource = QSharedPointer<QGeoPositionInfoSource>(QGeoPositionInfoSource::createDefaultSource(this));
    if (m_geoPositionInfoSource.data() != 0) {
        QObject::connect(m_geoPositionInfoSource.data(), SIGNAL(positionUpdated(QGeoPositionInfo)), this, SLOT(positionUpdated(QGeoPositionInfo)));
        QObject::connect(m_geoPositionInfoSource.data(), SIGNAL(updateTimeout()), this, SLOT(updateTimeout()));
    }
}

void Geolocation::getCurrentPosition(int scId, int ecId, QVariantMap p_options) {
    Q_UNUSED(p_options)

    m_successCallbacks << scId;
    m_errorCallbacks << ecId;

    // Check if we have a valid source
    if (m_geoPositionInfoSource != 0) {
        m_geoPositionInfoSource->requestUpdate();
    }
    else {
        // TODO: Replace with correct error code
        this->updateTimeout();
    }
}

void Geolocation::positionUpdated(const QGeoPositionInfo &update) {
    QGeoCoordinate coordinate = update.coordinate();

    auto t = std::make_tuple(coordinate.longitude(), coordinate.altitude(), update.attribute(QGeoPositionInfo::HorizontalAccuracy),
                             update.attribute(QGeoPositionInfo::VerticalAccuracy), update.attribute(QGeoPositionInfo::Direction), 
                             update.attribute(QGeoPositionInfo::GroundSpeed), update.timestamp().toMSecsSinceEpoch());

    QString callbackArguments = QString("Position.cast(Coordinates.cast(%1))").arg(CordovaInternal::tuple2str(t));

    for (int i = 0; i < m_successCallbacks.size(); i++) {
        this->callback(m_successCallbacks.at(i), callbackArguments);
    }

    m_errorCallbacks.clear();
    m_successCallbacks.clear();
}

void Geolocation::updateTimeout() {
    for (int i = 0; i < m_errorCallbacks.size(); i++) {
        this->callback(m_errorCallbacks.at(i), "PositionError.cast(PositionError.TIMEOUT, 'Position request timed out.')");
    }

    m_errorCallbacks.clear();
    m_successCallbacks.clear();
}
