;; tex-send-fax.el
;; send a latex buffer as fax
;; (C) Harald Milz (hm@seneca.ix.de) 1995
;; this is placed under the GNU Public License 2.0 (or newer).

(defun tex-send-fax ()
"Send the current LaTeX buffer as fax, using sendfax/faxspool from
Gert Doering's mgetty+sendfax package.  

You must TeX the buffer first to have the *.dvi file.

Additional pages (appendices) can be appended with the statement

%APPEND \"file1\" 'file2' ...

starting at column 0 of an arbitrary line. This is a comment to (thus
being ignored by) LaTeX and is being detected by tex-send-fax.
Appended files must be readable by faxspool. They are not printed by
tex-send-fax-print and not previewed by tex-send-fax-preview
(primarily because arbitrary file formats would have to be converted
into postscript... I'm lazy). The %APPEND statement can be deactivated
by prepending a second percent sign or whitespace. You can use either
\" or ' to protect shell metacharacters from being interpreted. \" has
the advantage of allowing for shell variables like $HOME to be
interpreted. Appendices can be in any of the formats that faxspool
converts automagically (see the faxspool man page and source),
e.g. DVI, PS, TIF, GIF etc.

tex-send-fax uses the tex-send-fax-faxspool variable to point to your 
faxspool command. You might want to 
(setq tex-send-fax-faxspool \"<faxspool-path>\") 
in your tex-mode-hook.el. 

The fax number is detected from a line starting with \"Fax-Nr.:\". A
good place for that line is the address paragraph, e.g.

\begin{letter}{
Mr. Example\\
1234 Example Rd.\\
Example, EX 12345\par
Fax-Nr.: 1 (415) 123 1231
}

Keys: C-c C-x
"
  (interactive)
  (save-excursion
    (beginning-of-buffer)
    ;; find addressee (and allow for carriage returns for 
    ;; formatting convenience)
    (while (re-search-forward "begin{letter}{[
]*" nil t) 
      (progn
	(end-of-line)
	(forward-char -2)
	(setq tex-send-fax-to (buffer-substring (match-end 0) (point)))))
    ;; find fax number 
    (while (re-search-forward "^Fax-Nr.:" nil t)
      (progn 
	(end-of-line)
	(setq tex-send-fax-number (buffer-substring (match-end 0) (point)))))
    ;; find files to fax...
    ;; ... at first the DVI file...
    (setq tex-send-fax-dvi-file (concat tex-print-file ".dvi"))
    ;; ... followed by the appendices, if any.
    (goto-char (point-min))
    (setq tex-send-fax-append " ")
    (while (re-search-forward "^%APPEND[ ]*" nil t)
      (progn
	(end-of-line)
        ;; FIXME: we should place a sanity check here: are all the
        ;; files to be appended readable?
	(setq tex-send-fax-append (buffer-substring (match-end 0) (point)))))
    ;; this is unelegant too: delete everything but numbers 
    ;; from the fax number
    (set-buffer "*scratch*")
    (erase-buffer)
    (insert tex-send-fax-number)
    (beginning-of-buffer)
    (replace-regexp "[^0-9]" "")
    (setq tex-send-fax-number (buffer-substring (point-min) (point-max)))
    (erase-buffer))
  (if (file-readable-p tex-send-fax-dvi-file)
      (progn  
	;; compose command line for faxspool
	(setq tex-send-fax-command 
              (concat 
               tex-send-fax-faxspool 
               " -D '" tex-send-fax-to "' " 
               tex-send-fax-number 
               " '" tex-send-fax-dvi-file "' " 
               tex-send-fax-append))
	(tex-send-fax-issue-command tex-send-fax-command))
    (message "file not found: %s" tex-send-fax-dvi-file))
  (message "append: %s" tex-send-fax-append))


;; preview a fax with dvips and ghostview

(defun tex-send-fax-preview ()
"Preview a fax with dvips and ghostview. xdvi cannot handle
EPSF \special statements. An external shell command is used:
/usr/local/bin/tex-send-fax-preview

Appended files are not shown (see tex-send-fax).

Keys: C-c C-y
"
  (interactive)
  (setq tex-send-fax-dvi-file (concat tex-print-file ".dvi"))
  (if (file-readable-p tex-send-fax-dvi-file)
      (progn
	(setq tex-send-fax-preview-command 
	      (concat 
	       "/usr/local/bin/tex-send-fax-preview " 
	       tex-send-fax-dvi-file))
	(tex-send-fax-issue-command tex-send-fax-preview-command))
    (message "file not found: %s" tex-send-fax-dvi-file)))


;; print a fax with dvips and ghostscript

(defun tex-send-fax-print ()
"Print a fax with dvips and ghostscript. dvilj2 cannot handle
EPSF \special statements. An external shell command is used:
/usr/local/bin/tex-send-fax-print

Appended files are not printed (see tex-send-fax).

Keys: C-c C-n
"
  (interactive)
  (if (file-readable-p tex-send-fax-dvi-file)
      (progn
	(setq tex-send-fax-print-command 
	      (concat 
	       "/usr/local/bin/tex-send-fax-print " 
	       tex-send-fax-dvi-file))
	(tex-send-fax-issue-command tex-send-fax-print-command))
    (message "file not found: %s" tex-send-fax-dvi-file)))


;; issue a command to the inferior shell
  
(defun tex-send-fax-issue-command (command)
"issue command"
  (save-excursion
    ;; there probably is a more elegant (and correct) way to 
    ;; do this, but it seems to work ...
    (set-buffer "*tex-shell*")
    (goto-char (point-max))
    (setq comint-process-echoes nil)
    (setq comint-eol-on-send t)
    (insert command)
    (goto-char (point-max))
    (comint-send-input)))
    
