<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Symfony\Component\Security\Core\User;

use Symfony\Component\Security\Core\Exception\UnsupportedUserException;
use Symfony\Component\Security\Core\Exception\UserNotFoundException;

/**
 * Chain User Provider.
 *
 * This provider calls several leaf providers in a chain until one is able to
 * handle the request.
 *
 * @author Johannes M. Schmitt <schmittjoh@gmail.com>
 *
 * @template-implements UserProviderInterface<UserInterface>
 */
class ChainUserProvider implements UserProviderInterface, PasswordUpgraderInterface
{
    /**
     * @param iterable<array-key, UserProviderInterface> $providers
     */
    public function __construct(
        private iterable $providers,
    ) {
    }

    /**
     * @return UserProviderInterface[]
     */
    public function getProviders(): array
    {
        if ($this->providers instanceof \Traversable) {
            return iterator_to_array($this->providers);
        }

        return $this->providers;
    }

    /**
     * @param array $attributes
     */
    public function loadUserByIdentifier(string $identifier/* , array $attributes = [] */): UserInterface
    {
        $attributes = \func_num_args() > 1 ? func_get_arg(1) : [];
        foreach ($this->providers as $provider) {
            try {
                if ($provider instanceof AttributesBasedUserProviderInterface || $provider instanceof self) {
                    return $provider->loadUserByIdentifier($identifier, $attributes);
                }

                return $provider->loadUserByIdentifier($identifier);
            } catch (UserNotFoundException) {
                // try next one
            }
        }

        $ex = new UserNotFoundException(\sprintf('There is no user with identifier "%s".', $identifier));
        $ex->setUserIdentifier($identifier);
        throw $ex;
    }

    public function refreshUser(UserInterface $user): UserInterface
    {
        $supportedUserFound = false;

        foreach ($this->providers as $provider) {
            try {
                if (!$provider->supportsClass(get_debug_type($user))) {
                    continue;
                }

                return $provider->refreshUser($user);
            } catch (UnsupportedUserException) {
                // try next one
            } catch (UserNotFoundException) {
                $supportedUserFound = true;
                // try next one
            }
        }

        if ($supportedUserFound) {
            $username = $user->getUserIdentifier();
            $e = new UserNotFoundException(\sprintf('There is no user with name "%s".', $username));
            $e->setUserIdentifier($username);
            throw $e;
        }

        throw new UnsupportedUserException(\sprintf('There is no user provider for user "%s". Shouldn\'t the "supportsClass()" method of your user provider return true for this classname?', get_debug_type($user)));
    }

    public function supportsClass(string $class): bool
    {
        foreach ($this->providers as $provider) {
            if ($provider->supportsClass($class)) {
                return true;
            }
        }

        return false;
    }

    public function upgradePassword(PasswordAuthenticatedUserInterface $user, string $newHashedPassword): void
    {
        foreach ($this->providers as $provider) {
            if ($provider instanceof PasswordUpgraderInterface) {
                try {
                    $provider->upgradePassword($user, $newHashedPassword);
                } catch (UnsupportedUserException) {
                    // ignore: password upgrades are opportunistic
                }
            }
        }
    }
}
