/*
    =============================================================================
    *****************************************************************************
    This Source Code Form is subject to the terms of the Mozilla Public
    License, v. 2.0. If a copy of the MPL was not distributed with this
    file, You can obtain one at https://mozilla.org/MPL/2.0/.

    Software distributed under the License is distributed on an "AS IS"
    basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
    License for the specific language governing rights and limitations
    under the License.

    The Original Code is jsAvroPhonetic

    The Initial Developer of the Original Code is
    Mehdi Hasan Khan <mhasan@omicronlab.com>
    Rifat Nabi <to.rifat@gmail.com>

    Copyright (C) OmicronLab (http://www.omicronlab.com). All Rights Reserved.


    Contributor(s): ______________________________________.

    *****************************************************************************
    =============================================================================
*/

const utfconv = imports.utf8;

function AvroRegex () {}

AvroRegex.prototype = {
    parse: function(input) {
        var fixed = this._fixString(input);
        var output = "";
        for(var cur = 0; cur < fixed.length; ++cur) {
            var start = cur, end = cur + 1, prev = start - 1;
            var matched = false;

            for(var i = 0; i < this.data.patterns.length; ++i) {
                var pattern = this.data.patterns[i];
                end = cur + pattern.find.length;
                if(end <= fixed.length && fixed.substring(start, end) == pattern.find) {
                    prev = start - 1;
                    if(typeof pattern.rules !== 'undefined') {
                        for(var j = 0; j < pattern.rules.length; ++j) {
                            var rule = pattern.rules[j];
                            var replace = true;

                            var chk = 0;
                            for(var k=0; k < rule.matches.length; ++k) {
                                var match = rule.matches[k];

                                if(match.type === "suffix") {
                                    chk = end;
                                } 
                                // Prefix
                                else {
                                    chk = prev;
                                }

                                // Handle Negative
                                if(typeof match.negative === 'undefined') {
                                    match.negative = false;
                                    if(match.scope.charAt(0) === '!') {
                                        match.negative = true;
                                        match.scope = match.scope.substring(1);
                                    }
                                }

                                // Handle empty value
                                if(typeof match.value === 'undefined') match.value = '';

                                // Beginning
                                if(match.scope === "punctuation") {
                                    if(
                                        ! (
                                            ((chk < 0) && (match.type === "prefix")) || 
                                            ((chk >= fixed.length) && (match.type === "suffix")) || 
                                            this._isPunctuation(fixed.charAt(chk))
                                        ) ^ match.negative
                                    ) {
                                        replace = false;
                                        break;
                                    }
                                }
                                // Vowel
                                else if(match.scope === "vowel") {
                                    if(
                                        ! (
                                            (
                                                (chk >= 0 && (match.type === "prefix")) || 
                                                (chk < fixed.length && (match.type === "suffix"))
                                            ) && 
                                            this._isVowel(fixed.charAt(chk))
                                        ) ^ match.negative
                                    ) {
                                        replace = false;
                                        break;
                                    }
                                }
                                // Consonant
                                else if(match.scope === "consonant") {
                                    if(
                                        ! (
                                            (
                                                (chk >= 0 && (match.type === "prefix")) || 
                                                (chk < fixed.length && match.type === ("suffix"))
                                            ) && 
                                            this._isConsonant(fixed.charAt(chk))
                                        ) ^ match.negative
                                    ) {
                                        replace = false;
                                        break;
                                    }
                                }
                                // Exact
                                else if(match.scope === "exact") {
                                    var s, e;
                                    if(match.type === "suffix") {
                                        s = end;
                                        e = end + match.value.length;
                                    } 
                                    // Prefix
                                    else {
                                        s = start - match.value.length;
                                        e = start;
                                    }
                                    if(!this._isExact(match.value, fixed, s, e, match.negative)) {
                                        replace = false;
                                        break;
                                    }
                                }
                            }

                            if(replace) {
                                output += rule.replace + "(্[যবম])?(্?)([ঃঁ]?)";
                                cur = end - 1;
                                matched = true;
                                break;
                            }

                        }
                    }
                    if(matched == true) break;

                    // Default
                    output += pattern.replace + "(্[যবম])?(্?)([ঃঁ]?)";
                    cur = end - 1;
                    matched = true;
                    break;
                }
            }

            if(!matched) {
                output += fixed.charAt(cur);
            }
        }
        return this._convertToUnicodeValue(output);
    },
    
    
    _convertToUnicodeValue: function(input){
        var output = '';
        input = utfconv.utf8Decode(input);
        
        for (var i = 0; i < input.length; i++){
            var charCode = input.charCodeAt(i);
            if (charCode >= 255){
                output += '\\u0' + charCode.toString(16);
            } else {
                output += input.charAt(i);
            }
        }
        return output;
    },
    
    
    _fixString: function(input) {
        var fixed = '';
        for(var i=0; i < input.length; ++i) {
            var cChar = input.charAt(i);
            if(!this._isIgnore(cChar)) {
                fixed += cChar.toLowerCase();
            }
        }
        return fixed;
    },
    
    
    _isVowel: function(c) {
        return (this.data.vowel.indexOf(c.toLowerCase()) >= 0);
    },
    
    
    _isConsonant: function(c) {
        return (this.data.consonant.indexOf(c.toLowerCase()) >= 0);
    },
    
    
    _isPunctuation: function(c) {
        return (!(this._isVowel(c) || this._isConsonant(c)));
    },
    
    
    _isExact: function(needle, heystack, start, end, not) {
        return ((start >= 0 && end < heystack.length && (heystack.substring(start, end)  === needle)) ^ not);
    },
    
    
    _isIgnore: function(c) {
        return (this.data.ignore.indexOf(c.toLowerCase()) >= 0);
    },
    
    
    data: {
        "patterns":
            [
                {
                    "find":"aa",
                    "replace":"(আ|(য়া)|া|((([অএ]্যা?)|[আএ]|([‍‌]?(্য)?া)|(য়া))((([অএ]্যা?)|[আএ]|([‍‌]?(্য)?া)|(য়া)))?))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"au",
                    "replace":"(ঔ|ৌ(([অএ]্যা?)|[আএ]|([‍‌]?(্য)?া)|(য়া))([উঊুূ]|(য়[ুূ])))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"az",
                    "replace":"((([অএ]্যা?)|[আএ]|([‍‌]?(্য)?া)|(য়া))((জ|য|(জ়)|([‍‌]?্য)))?)",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ai",
                    "replace":"(ঐ|ৈ|(([অএ]্যা?)|[আএ]|([‍‌]?(্য)?া)|(য়া))([ইঈিী]|(য়[িী])))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"a",
                    "replace":"(([অএ]্যা?)|[অআএ]|([‍‌]?(্য)?া)|(য়া))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"bdh",
                    "replace":"((ব(্?)(ধ|ঢ|([দড](্?)(হ|ঃ|(হ্‌?)))))|((ব(্?)[দড])(্?)(হ|ঃ|(হ্‌?)))|(ব(্?)[দড](্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"bhl",
                    "replace":"(((ভ|(ব(্?)(হ|ঃ|(হ্‌?))))(্?)ল)|(ব(্?)((হ|ঃ|(হ্‌?))(্?)ল))|(ব(্?)(হ|ঃ|(হ্‌?))(্?)ল))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"bh",
                    "replace":"(ভ|(ব(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"bb",
                    "replace":"(ব(্?)ব?)",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"bd",
                    "replace":"(ব(্?)[দড])",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"bv",
                    "replace":"(ব?(্?)ভ)",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"b",
                    "replace":"ব",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"chchh",
                    "replace":"((চ্ছ)|((চ|ছ|([চছ](্?)(হ|ঃ|(হ্‌?))))(্?)(((চ|ছ|([চছ](্?)(হ|ঃ|(হ্‌?))))(্?)((হ|ঃ|(হ্‌?)))?)|([চছ](্?)((হ|ঃ|(হ্‌?))(্?)(হ|ঃ|(হ্‌?))))|([চছ](্?)(হ|ঃ|(হ্‌?))(্?)(হ|ঃ|(হ্‌?)))))|((চ|ছ|([চছ](্?)(হ|ঃ|(হ্‌?))))(্?)(চ|ছ|([চছ](্?)(হ|ঃ|(হ্‌?))))(্?)(হ|ঃ|(হ্‌?)))|([চছ](্?)(হ|ঃ|(হ্‌?))(্?)[চছ](্?)(হ|ঃ|(হ্‌?))(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"cch",
                    "replace":"(([চছ](্?)(চ|ছ|([চছ](্?)(হ|ঃ|(হ্‌?)))))|([চছ](্?)[চছ](্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"chh",
                    "replace":"(((চ|ছ|([চছ](্?)(হ|ঃ|(হ্‌?))))(্?)((হ|ঃ|(হ্‌?)))?)|([চছ](্?)((হ|ঃ|(হ্‌?))(্?)(হ|ঃ|(হ্‌?))))|([চছ](্?)(হ|ঃ|(হ্‌?))(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"cng",
                    "replace":"((চ্ঞ)|([চছ](্?)(ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ)))))|([চছ](্?)[নণঁঙঞং](্?)(গ|(জ্ঞ)))|(([চছ](্?)[নণঁঙঞং])(্?)(গ|(জ্ঞ))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ch",
                    "replace":"(চ|ছ|([চছ](্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ck",
                    "replace":"(ক|([চছ](্?)ক))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"cc",
                    "replace":"([চছ](্?)[চছ]?)",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"cn",
                    "replace":"([চছ](্?)[নণঁঙঞং])",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"c",
                    "replace":"[চছ]",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"dhm",
                    "replace":"((([দড](্?)(গ|(জ্ঞ)))(্?)ম)|([দড](্?)((হ|ঃ|(হ্‌?))(্?)ম))|([দড](্?)(হ|ঃ|(হ্‌?))(্?)ম))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"dhn",
                    "replace":"(((ধ|ঢ|([দড](্?)(হ|ঃ|(হ্‌?))))(্?)[নণঁঙঞং])|([দড](্?)((হ|ঃ|(হ্‌?))(্?)[নণঁঙঞং]))|([দড](্?)(হ|ঃ|(হ্‌?))(্?)[নণঁঙঞং]))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"dbh",
                    "replace":"(([দড](্?)(ভ|(ব(্?)(হ|ঃ|(হ্‌?)))))|(([দড](্?)ব)(্?)(হ|ঃ|(হ্‌?)))|([দড](্?)ব(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ddh",
                    "replace":"(([দড]?(্?)([দড](্?)(গ|(জ্ঞ))))|(([দড](্?)[দড]?)(্?)(হ|ঃ|(হ্‌?)))|([দড](্?)[দড](্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"dgh",
                    "replace":"(([দড](্?)(ঘ|((গ|(জ্ঞ))(্?)(হ|ঃ|(হ্‌?)))))|(([দড](্?)(গ|(জ্ঞ)))(্?)(হ|ঃ|(হ্‌?)))|([দড](্?)(গ|(জ্ঞ))(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"db",
                    "replace":"([দড](্?)ব)",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"dd",
                    "replace":"([দড](্?)[দড]?)",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"dg",
                    "replace":"([দড](্?)(গ|(জ্ঞ)))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"dh",
                    "replace":"(ধ|ঢ|([দড](্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"d",
                    "replace":"[দড]",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ey",
                    "replace":"(এ|ই|ে|(েই)|(এই)|ঈ|ী|(((এ্যা?)|[এে]|([‍‌]?(্য)া)|(য়ে))(য়|(ইয়)|([‍‌]?্য))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ee",
                    "replace":"(ই|ঈ|ি|ী|(য়েই)|(((এ্যা?)|[এে]|([‍‌]?(্য)া)|(য়ে))((এ্যা?)|[এে]|([‍‌]?(্য)া)|(য়ে))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"e",
                    "replace":"((এ্যা?)|[এে]|([‍‌]?(্য)া)|(য়ে))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ff",
                    "replace":"(ফ(্?)ফ?)",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"f",
                    "replace":"ফ",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ggh",
                    "replace":"((((জ্ঞ)|((গ|(জ্ঞ))(্?)((গ|(জ্ঞ)))?))(্?)(হ|ঃ|(হ্‌?)))|((গ|(জ্ঞ))(্?)(ঘ|((গ|(জ্ঞ))(্?)(হ|ঃ|(হ্‌?)))))|((গ|(জ্ঞ))(্?)(গ|(জ্ঞ))(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"gdh",
                    "replace":"(((গ|(জ্ঞ))(্?)(ধ|ঢ|([দড](্?)(হ|ঃ|(হ্‌?)))))|((গ|(জ্ঞ))(্?)[দড](্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ghn",
                    "replace":"(((ঘ|((গ|(জ্ঞ))(্?)(হ|ঃ|(হ্‌?))))(্?)[নণঁঙঞং])|((গ|(জ্ঞ))(্?)((হ|ঃ|(হ্‌?))(্?)[নণঁঙঞং]))|((গ|(জ্ঞ))(্?)(হ|ঃ|(হ্‌?))(্?)[নণঁঙঞং]))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"gh",
                    "replace":"(ঘ|((গ|(জ্ঞ))(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"gg",
                    "replace":"((জ্ঞ)|((গ|(জ্ঞ))(্?)((গ|(জ্ঞ)))?))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"g",
                    "replace":"(গ|(জ্ঞ))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"hl",
                    "replace":"((হ|ঃ|(হ্‌?))(্?)ল)",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"hh",
                    "replace":"((হ|ঃ|(হ্‌?))(্?)(হ|ঃ|(হ্‌?)))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"hm",
                    "replace":"((হ|ঃ|(হ্‌?))(্?)ম)",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"hn",
                    "replace":"((হ|ঃ|(হ্‌?))(্?)[নণঁঙঞং])",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"h",
                    "replace":"(হ|ঃ|(হ্‌?))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ia",
                    "replace":"((ঞা)|(([ইঈিী]|(য়[িী]))(([অএ]্যা?)|[আএ]|([‍‌]?(্য)?া)|(য়া))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"i",
                    "replace":"([ইঈিী]|(য়[িী]))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"jng",
                    "replace":"((জ্ঞ)|(([জয]|(জ়))(্?)(ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ)))))|(([জয]|(জ়))(্?)[নণঁঙঞং](্?)(গ|(জ্ঞ))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"jjh",
                    "replace":"(((([জয]|(জ়)))?(্?)(ঝ|(([জয]|(জ়))(্?)(হ|ঃ|(হ্‌?)))))|(হ্য)|(((হ্য)|(([জয]|(জ়))(্?)(([জয]|(জ়)))?))(্?)(হ|ঃ|(হ্‌?)))|(([জয]|(জ়))(্?)([জয]|(জ়))(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"jj",
                    "replace":"((হ্য)|(([জয]|(জ়))(্?)(([জয]|(জ়)))?))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"jh",
                    "replace":"(ঝ|(([জয]|(জ়))(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"j",
                    "replace":"([জয]|(জ়))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"kshm",
                    "replace":"((((ক(্?)(স|শ|ষ|([সশষ](্?)(হ|ঃ|(হ্‌?)))))|((ক(্?)[সশষ])(্?)(হ|ঃ|(হ্‌?)))|(ক(্?)[সশষ](্?)(হ|ঃ|(হ্‌?))))(্?)ম)|(ক(্?)(((স|শ|ষ|([সশষ](্?)(হ|ঃ|(হ্‌?))))(্?)ম)|([সশষ](্?)((হ|ঃ|(হ্‌?))(্?)ম))|([সশষ](্?)(হ|ঃ|(হ্‌?))(্?)ম)))|((ক(্?)[সশষ])(্?)((হ|ঃ|(হ্‌?))(্?)ম))|(ক(্?)(স|শ|ষ|([সশষ](্?)(হ|ঃ|(হ্‌?))))(্?)ম)|(ক(্?)[সশষ](্?)(হ|ঃ|(হ্‌?))(্?)ম))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"kkhm",
                    "replace":"((((ক্ষ)|((ক(্?)ক?)(্?)(হ|ঃ|(হ্‌?)))|(ক?(্?)(খ|(ক্ষ)|(ক(্?)(হ|ঃ|(হ্‌?)))))|(ক(্?)ক(্?)(হ|ঃ|(হ্‌?))))(্?)ম)|((ক(্?)ক?)(্?)((হ|ঃ|(হ্‌?))(্?)ম))|(ক(্?)ক(্?)(হ|ঃ|(হ্‌?))(্?)ম)|(ক(্?)(খ|(ক্ষ)|(ক(্?)(হ|ঃ|(হ্‌?))))(্?)ম))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"kshn",
                    "replace":"((((ক(্?)(স|শ|ষ|([সশষ](্?)(হ|ঃ|(হ্‌?)))))|((ক(্?)[সশষ])(্?)(হ|ঃ|(হ্‌?)))|(ক(্?)[সশষ](্?)(হ|ঃ|(হ্‌?))))(্?)[নণঁঙঞং])|(ক(্?)(((স|শ|ষ|([সশষ](্?)(হ|ঃ|(হ্‌?))))(্?)[নণঁঙঞং])|([সশষ](্?)((হ|ঃ|(হ্‌?))(্?)[নণঁঙঞং]))|([সশষ](্?)(হ|ঃ|(হ্‌?))(্?)[নণঁঙঞং])))|((ক(্?)[সশষ])(্?)((হ|ঃ|(হ্‌?))(্?)[নণঁঙঞং]))|(ক(্?)(স|শ|ষ|([সশষ](্?)(হ|ঃ|(হ্‌?))))(্?)[নণঁঙঞং])|(ক(্?)[সশষ](্?)(হ|ঃ|(হ্‌?))(্?)[নণঁঙঞং]))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"kkhn",
                    "replace":"((((ক্ষ)|((ক(্?)ক?)(্?)(হ|ঃ|(হ্‌?)))|(ক?(্?)(খ|(ক্ষ)|(ক(্?)(হ|ঃ|(হ্‌?)))))|(ক(্?)ক(্?)(হ|ঃ|(হ্‌?))))(্?)[নণঁঙঞং])|((ক(্?)ক?)(্?)((হ|ঃ|(হ্‌?))(্?)[নণঁঙঞং]))|(ক(্?)ক(্?)(হ|ঃ|(হ্‌?))(্?)[নণঁঙঞং])|(ক(্?)(খ|(ক্ষ)|(ক(্?)(হ|ঃ|(হ্‌?))))(্?)[নণঁঙঞং]))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ksh",
                    "replace":"((ক(্?)(স|শ|ষ|([সশষ](্?)(হ|ঃ|(হ্‌?)))))|((ক(্?)[সশষ])(্?)(হ|ঃ|(হ্‌?)))|(ক(্?)[সশষ](্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"kkh",
                    "replace":"((ক্ষ)|((ক(্?)ক?)(্?)(হ|ঃ|(হ্‌?)))|(ক?(্?)(খ|(ক্ষ)|(ক(্?)(হ|ঃ|(হ্‌?)))))|(ক(্?)ক(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"kxm",
                    "replace":"((((ক্ষ)|(ক(্?)((ক্স)|(এক্স)|ষ)))(্?)ম)|(ক(্?)(((ক্স)|(এক্স)|ষ)(্?)ম))|(ক(্?)((ক্স)|(এক্স)|ষ)(্?)ম))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"kxn",
                    "replace":"((((ক্ষ)|(ক(্?)((ক্স)|(এক্স)|ষ)))(্?)[নণঁঙঞং])|(ক(্?)(((ক্স)|(এক্স)|ষ)(্?)[নণঁঙঞং]))|(ক(্?)((ক্স)|(এক্স)|ষ)(্?)[নণঁঙঞং]))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"kh",
                    "replace":"(খ|(ক্ষ)|(ক(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"kk",
                    "replace":"(ক(্?)ক?)",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ks",
                    "replace":"(ক(্?)[সশষ])",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"kx",
                    "replace":"((ক্ষ)|(ক(্?)((ক্স)|(এক্স)|ষ)))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"k",
                    "replace":"ক",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"lkh",
                    "replace":"((ল(্?)(খ|(ক্ষ)|(ক(্?)(হ|ঃ|(হ্‌?)))))|((ল(্?)ক)(্?)(হ|ঃ|(হ্‌?)))|(ল(্?)ক(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"lgh",
                    "replace":"((ল(্?)(ঘ|((গ|(জ্ঞ))(্?)(হ|ঃ|(হ্‌?)))))|((ল(্?)(গ|(জ্ঞ)))(্?)(হ|ঃ|(হ্‌?)))|(ল(্?)(গ|(জ্ঞ))(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"lph",
                    "replace":"((ল(্?)(ফ|(প(্?)(হ|ঃ|(হ্‌?)))))|((ল(্?)প)(্?)(হ|ঃ|(হ্‌?)))|(ল(্?)প(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ldh",
                    "replace":"((ল(্?)(ধ|ঢ|([দড](্?)(হ|ঃ|(হ্‌?)))))|((ল(্?)[দড])(্?)(হ|ঃ|(হ্‌?)))|(ল(্?)[দড](্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"lbh",
                    "replace":"((ল(্?)(ভ|(ব(্?)(হ|ঃ|(হ্‌?)))))|((ল(্?)ব)(্?)(হ|ঃ|(হ্‌?)))|(ল(্?)ব(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ll",
                    "replace":"((হ্ল)|(ল?(্?)ল)|(ল(্?)ল))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"lk",
                    "replace":"(ল(্?)ক)",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"lg",
                    "replace":"(ল(্?)(গ|(জ্ঞ)))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"lp",
                    "replace":"(ল(্?)প)",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ld",
                    "replace":"(ল(্?)[দড])",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"lb",
                    "replace":"(ল(্?)ব)",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"l",
                    "replace":"ল",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"mbh",
                    "replace":"((ম(্?)(ভ|(ব(্?)(হ|ঃ|(হ্‌?)))))|((ম(্?)ব)(্?)(হ|ঃ|(হ্‌?)))|(ম(্?)ব(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"mph",
                    "replace":"((ম(্?)(ফ|(প(্?)(হ|ঃ|(হ্‌?)))))|((ম(্?)প)(্?)(হ|ঃ|(হ্‌?)))|(ম(্?)প(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"mth",
                    "replace":"((ম(্?)(থ|ঠ|([তটৎ](্?)(হ|ঃ|(হ্‌?)))))|((ম(্?)[তটৎ])(্?)(হ|ঃ|(হ্‌?)))|(ম(্?)[তটৎ](্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"mm",
                    "replace":"((হ্ম)|(ম(্?)ম?))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"mb",
                    "replace":"(ম(্?)ব)",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"mp",
                    "replace":"(ম(্?)প)",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"mt",
                    "replace":"(ম(্?)[তটৎ])",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"m",
                    "replace":"ম",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ngksh",
                    "replace":"(((((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)ক)|([নণঁঙঞং](্?)(গ|(জ্ঞ))(্?)ক))(্?)(স|শ|ষ|([সশষ](্?)(হ|ঃ|(হ্‌?)))))|((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)ক(্?)(স|শ|ষ|([সশষ](্?)(হ|ঃ|(হ্‌?)))))|((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)((ক(্?)(স|শ|ষ|([সশষ](্?)(হ|ঃ|(হ্‌?)))))|((ক(্?)[সশষ])(্?)(হ|ঃ|(হ্‌?)))|(ক(্?)[সশষ](্?)(হ|ঃ|(হ্‌?)))))|((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)(ক(্?)[সশষ])(্?)(হ|ঃ|(হ্‌?)))|([নণঁঙঞং](্?)(গ|(জ্ঞ))(্?)ক(্?)[সশষ](্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ngkkh",
                    "replace":"(((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)((ক্ষ)|((ক(্?)ক?)(্?)(হ|ঃ|(হ্‌?)))|(ক?(্?)(খ|(ক্ষ)|(ক(্?)(হ|ঃ|(হ্‌?)))))|(ক(্?)ক(্?)(হ|ঃ|(হ্‌?)))))|((((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)ক)|([নণঁঙঞং](্?)(গ|(জ্ঞ))(্?)ক))(্?)(খ|(ক্ষ)|(ক(্?)(হ|ঃ|(হ্‌?)))))|((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)ক(্?)(খ|(ক্ষ)|(ক(্?)(হ|ঃ|(হ্‌?)))))|((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)(ক(্?)ক?)(্?)(হ|ঃ|(হ্‌?)))|([নণঁঙঞং](্?)(গ|(জ্ঞ))(্?)ক(্?)ক(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ngch",
                    "replace":"(((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)(চ|ছ|([চছ](্?)(হ|ঃ|(হ্‌?)))))|((((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)[চছ])|([নণঁঙঞং](্?)(গ|(জ্ঞ))(্?)[চছ]))(্?)(হ|ঃ|(হ্‌?)))|([নণঁঙঞং](্?)(গ|(জ্ঞ))(্?)[চছ](্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"nggh",
                    "replace":"(((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)(ঘ|((গ|(জ্ঞ))(্?)(হ|ঃ|(হ্‌?)))))|([নণঁঙঞং](্?)((জ্ঞ)|((গ|(জ্ঞ))(্?)((গ|(জ্ঞ)))?))(্?)(হ|ঃ|(হ্‌?)))|([নণঁঙঞং](্?)((((জ্ঞ)|((গ|(জ্ঞ))(্?)((গ|(জ্ঞ)))?))(্?)(হ|ঃ|(হ্‌?)))|((গ|(জ্ঞ))(্?)(ঘ|((গ|(জ্ঞ))(্?)(হ|ঃ|(হ্‌?)))))|((গ|(জ্ঞ))(্?)(গ|(জ্ঞ))(্?)(হ|ঃ|(হ্‌?)))))|([নণঁঙঞং](্?)(গ|(জ্ঞ))(্?)(গ|(জ্ঞ))(্?)(হ|ঃ|(হ্‌?)))|((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)(গ|(জ্ঞ))(্?)(হ|ঃ|(হ্‌?)))|(((ঙ্গ)|((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)(গ|(জ্ঞ)))|([নণঁঙঞং](্?)(গ|(জ্ঞ))(্?)(গ|(জ্ঞ))))(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ngkx",
                    "replace":"(((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)((ক্ষ)|(ক(্?)((ক্স)|(এক্স)|ষ))))|((((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)ক)|([নণঁঙঞং](্?)(গ|(জ্ঞ))(্?)ক))(্?)((ক্স)|(এক্স)|ষ))|([নণঁঙঞং](্?)(গ|(জ্ঞ))(্?)((ক্ষ)|(ক(্?)((ক্স)|(এক্স)|ষ))))|([নণঁঙঞং](্?)(গ|(জ্ঞ))(্?)ক(্?)((ক্স)|(এক্স)|ষ)))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ngjh",
                    "replace":"(((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)(ঝ|(([জয]|(জ়))(্?)(হ|ঃ|(হ্‌?)))))|((((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)([জয]|(জ়)))|([নণঁঙঞং](্?)(গ|(জ্ঞ))(্?)([জয]|(জ়))))(্?)(হ|ঃ|(হ্‌?)))|([নণঁঙঞং](্?)(গ|(জ্ঞ))(্?)([জয]|(জ়))(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ngkh",
                    "replace":"(((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)(খ|(ক্ষ)|(ক(্?)(হ|ঃ|(হ্‌?)))))|((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)ক(্?)(হ|ঃ|(হ্‌?)))|([নণঁঙঞং](্?)(গ|(জ্ঞ))(্?)ক(্?)(হ|ঃ|(হ্‌?)))|((((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)ক)|([নণঁঙঞং](্?)(গ|(জ্ঞ))(্?)ক))(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"nsh",
                    "replace":"(([নণঁঙঞং](্?)(স|শ|ষ|([সশষ](্?)(হ|ঃ|(হ্‌?)))))|([নণঁঙঞং](্?)[সশষ](্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ndh",
                    "replace":"(([নণঁঙঞং](্?)(ধ|ঢ|([দড](্?)(হ|ঃ|(হ্‌?)))))|(([নণঁঙঞং](্?)[দড])(্?)(হ|ঃ|(হ্‌?)))|([নণঁঙঞং](্?)[দড](্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"nkh",
                    "replace":"(([নণঁঙঞং](্?)(খ|(ক্ষ)|(ক(্?)(হ|ঃ|(হ্‌?)))))|(((ঙ্ক)|([নণঁঙঞং](্?)ক))(্?)(হ|ঃ|(হ্‌?)))|([নণঁঙঞং](্?)ক(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"nth",
                    "replace":"(([নণঁঙঞং](্?)(থ|ঠ|([তটৎ](্?)(হ|ঃ|(হ্‌?)))))|(([নণঁঙঞং](্?)[তটৎ])(্?)(হ|ঃ|(হ্‌?)))|([নণঁঙঞং](্?)[তটৎ](্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ngj",
                    "replace":"(((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)([জয]|(জ়)))|([নণঁঙঞং](্?)(গ|(জ্ঞ))(্?)([জয]|(জ়))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ngm",
                    "replace":"(((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)ম)|([নণঁঙঞং](্?)(গ|(জ্ঞ))(্?)ম))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ngg",
                    "replace":"((ঙ্গ)|((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)(গ|(জ্ঞ)))|([নণঁঙঞং](্?)(গ|(জ্ঞ))(্?)(গ|(জ্ঞ))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ngx",
                    "replace":"(((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)((ক্স)|(এক্স)|ষ))|([নণঁঙঞং](্?)(গ|(জ্ঞ))(্?)((ক্স)|(এক্স)|ষ)))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ngk",
                    "replace":"(((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)ক)|([নণঁঙঞং](্?)(গ|(জ্ঞ))(্?)ক))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ngh",
                    "replace":"((ঙ্ঘ)|([নণঁঙঞং](্?)(ঘ|((গ|(জ্ঞ))(্?)(হ|ঃ|(হ্‌?)))))|([নণঁঙঞং](্?)(গ|(জ্ঞ))(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"nch",
                    "replace":"(([নণঁঙঞং](্?)(চ|ছ|([চছ](্?)(হ|ঃ|(হ্‌?)))))|(((ঞ্চ)|([নণঁঙঞং](্?)[চছ]))(্?)(হ|ঃ|(হ্‌?)))|([নণঁঙঞং](্?)[চছ](্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"njh",
                    "replace":"(([নণঁঙঞং](্?)(ঝ|(([জয]|(জ়))(্?)(হ|ঃ|(হ্‌?)))))|(((ঞ্জ)|([নণঁঙঞং](্?)([জয]|(জ়))))(্?)(হ|ঃ|(হ্‌?)))|([নণঁঙঞং](্?)([জয]|(জ়))(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ngc",
                    "replace":"(((ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))(্?)[চছ])|([নণঁঙঞং](্?)(গ|(জ্ঞ))(্?)[চছ]))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"nc",
                    "replace":"((ঞ্চ)|([নণঁঙঞং](্?)[চছ]))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"nn",
                    "replace":"((হ্ণ)|(হ্ন)|([নণঁঙঞং](্?)[নণঁঙঞং]?))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ng",
                    "replace":"(ঙ|ং|ঞ|(ঙ্গ)|([নণঁঙঞং](্?)(গ|(জ্ঞ))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"nk",
                    "replace":"((ঙ্ক)|([নণঁঙঞং](্?)ক))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"nj",
                    "replace":"((ঞ্জ)|([নণঁঙঞং](্?)([জয]|(জ়))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"nd",
                    "replace":"([নণঁঙঞং](্?)[দড])",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"nt",
                    "replace":"([নণঁঙঞং](্?)[তটৎ])",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"n",
                    "replace":"[নণঁঙঞং]",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"oo",
                    "replace":"((([উঊুূ]|(য়[ুূ])))|(([ওোঅ]|(অ্য)|(য়ো?))?([ওোঅ]|(অ্য)|(য়ো?))?))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"oi",
                    "replace":"(ঐ|ৈ|(([ওোঅ]|(অ্য)|(য়ো?))?([ইঈিী]|(য়[িী]))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ou",
                    "replace":"(ঔ|ৌ|(([ওোঅ]|(অ্য)|(য়ো?))?([উঊুূ]|(য়[ুূ]))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"o",
                    "replace":"([ওোঅ]|(অ্য)|(য়ো?))?",
                    "rules":
                    [
                        {
                            "matches":
                            [
                                {
                                    "type":"prefix",
                                    "scope":"punctuation",
                                    "value":"",
                                    "negative":"FALSE"
                                }
                            ],
                            "replace":"([ওোঅ]|(অ্য)|(য়ো?))"
                        }
                    ]
                },
                {
                    "find":"phl",
                    "replace":"(((ফ|(প(্?)(হ|ঃ|(হ্‌?))))(্?)ল)|(প(্?)((হ|ঃ|(হ্‌?))(্?)ল))|(প(্?)(হ|ঃ|(হ্‌?))(্?)ল))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ph",
                    "replace":"(ফ|(প(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"pp",
                    "replace":"(প(্?)প?)",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"p",
                    "replace":"প",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"qq",
                    "replace":"(ক(্?)ক?)",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"q",
                    "replace":"ক",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"rri",
                    "replace":"(ঋ|ৃ|(([রড়ঢ়]|(হ্র))([রড়ঢ়]|(হ্র))([ইঈিী]|(য়[িী]))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ri",
                    "replace":"(ঋ|ৃ|(হৃ)|(([রড়ঢ়]|(হ্র))([ইঈিী]|(য়[িী]))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"rh",
                    "replace":"((([রড়ঢ়]|(হ্র)))|(([রড়ঢ়]|(হ্র))(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"r",
                    "replace":"([রড়ঢ়]|(হ্র))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"shsh",
                    "replace":"((((স|শ|ষ|([সশষ](্?)(হ|ঃ|(হ্‌?)))))?(্?)(স|শ|ষ|([সশষ](্?)(হ|ঃ|(হ্‌?)))))|([সশষ](্?)(হ|ঃ|(হ্‌?))(্?)[সশষ](্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ssh",
                    "replace":"(([সশষ]?(্?)(স|শ|ষ|([সশষ](্?)(হ|ঃ|(হ্‌?)))))|([সশষ](্?)[সশষ](্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"shm",
                    "replace":"(((স|শ|ষ|([সশষ](্?)(হ|ঃ|(হ্‌?))))(্?)ম)|([সশষ](্?)((হ|ঃ|(হ্‌?))(্?)ম))|([সশষ](্?)(হ|ঃ|(হ্‌?))(্?)ম))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"shn",
                    "replace":"(((স|শ|ষ|([সশষ](্?)(হ|ঃ|(হ্‌?))))(্?)[নণঁঙঞং])|([সশষ](্?)((হ|ঃ|(হ্‌?))(্?)[নণঁঙঞং]))|([সশষ](্?)(হ|ঃ|(হ্‌?))(্?)[নণঁঙঞং]))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"ss",
                    "replace":"([সশষ](্?)[সশষ]?)",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"sh",
                    "replace":"(স|শ|ষ|([সশষ](্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"s",
                    "replace":"[সশষ]",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"thth",
                    "replace":"((ত্থ)|(((থ|ঠ|([তটৎ](্?)(হ|ঃ|(হ্‌?)))))?(্?)(থ|ঠ|([তটৎ](্?)(হ|ঃ|(হ্‌?)))))|([তটৎ](্?)(হ|ঃ|(হ্‌?))(্?)[তটৎ](্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"tth",
                    "replace":"(([তটৎ]?(্?)(থ|ঠ|([তটৎ](্?)(হ|ঃ|(হ্‌?)))))|(([তটৎ](্?)[তটৎ]?)(্?)(হ|ঃ|(হ্‌?)))|([তটৎ](্?)[তটৎ](্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"t``",
                    "replace":"ৎ",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"tth",
                    "replace":"(([তটৎ]?(্?)(থ|ঠ|([তটৎ](্?)(হ|ঃ|(হ্‌?)))))|(([তটৎ](্?)[তটৎ]?)(্?)(হ|ঃ|(হ্‌?)))|([তটৎ](্?)[তটৎ](্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"tt",
                    "replace":"([তটৎ](্?)[তটৎ]?)",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"th",
                    "replace":"(থ|ঠ|([তটৎ](্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"t",
                    "replace":"[তটৎ]",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"uu",
                    "replace":"(ঊ|ূ|(([উঊুূ]|(য়[ুূ]))(([উঊুূ]|(য়[ুূ])))?))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"u",
                    "replace":"([উঊুূ]|(য়[ুূ]))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"vv",
                    "replace":"(ভ(্?)ভ?)",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"v",
                    "replace":"ভ",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"w",
                    "replace":"(ও|(ওয়)|(্ব))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"xm",
                    "replace":"(((ক্স)|(এক্স)|ষ)(্?)ম)",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"xn",
                    "replace":"(((ক্স)|(এক্স)|ষ)(্?)[নণঁঙঞং])",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"x",
                    "replace":"((ক্স)|(এক্স)|ষ)",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"y",
                    "replace":"(য়|(ইয়)|([‍‌]?্য))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"zzh",
                    "replace":"((হ্য)|((জ|য|(জ়)|([‍‌]?্য))(্?)(ঝ|(([জয]|(জ়))(্?)(হ|ঃ|(হ্‌?)))))|(((হ্য)|((জ|য|(জ়)|([‍‌]?্য))(্?)((জ|য|(জ়)|([‍‌]?্য)))?))(্?)(হ|ঃ|(হ্‌?)))|((জ|য|(জ়)|([‍‌]?্য))(্?)(জ|য|(জ়)|([‍‌]?্য))(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"zh",
                    "replace":"(ঝ|(([জয]|(জ়))(্?)(হ|ঃ|(হ্‌?))))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"zz",
                    "replace":"((হ্য)|((জ|য|(জ়)|([‍‌]?্য))(্?)((জ|য|(জ়)|([‍‌]?্য)))?))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"z",
                    "replace":"(জ|য|(জ়)|([‍‌]?্য))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"0",
                    "replace":"(০|(0)|(শূন্য))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"1",
                    "replace":"(১|(1)|(এক))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"2",
                    "replace":"(২|(2)|(দুই))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"3",
                    "replace":"(৩|(3)|(তিন))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"4",
                    "replace":"(৪|(4)|(চার))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"5",
                    "replace":"(৫|(5)|(পাঁচ))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"6",
                    "replace":"((6)|৬|(ছয়))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"7",
                    "replace":"(৭|(7)|(সাত))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"8",
                    "replace":"(৮|(8)|(আট))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"9",
                    "replace":"(৯|(9)|(নয়))",
                    "rules":
                    [
                    ]
                },
                {
                    "find":"`",
                    "replace":"",
                    "rules":
                    [
                    ]
                }
            ],
            "vowel":"aeiou",
            "consonant":"bcdfghjklmnpqrstvwxyz",
            "ignore":"|()[]{}^$*+?.~!@#%&-_='\";<>/\\,:`"
    }
}

/* --------- */
/* Test code */
/* --------- */
//var __avroRegex = new AvroRegex ();
//print(__avroRegex.parse('a(!k)'));