/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iconfigure.h"
#if ISHELL_INCLUDED(ISHELL_GG)


#include "iggwidgetkeytrackball.h"


#include "icontrolmodule.h"
#include "imath.h"
#include "ishell.h"
#include "ivtk.h"

#include "iggframe.h"
#include "iggrenderwindowobserver.h"
#include "iggwidgetrendermodebutton.h"

#include "ibgwidgettrackballsubject.h"

#include <vtkMath.h>

#include "iggsubjectfactory.h"

//
//  templates
//
#include "iggwidgetkeyhandlertemplate.h"


using namespace iParameter;


//
// Constructs a TrackBall.
//
iggWidgetKeyTrackBall::iggWidgetKeyTrackBall(const iObjectKey &dirkey, bool followCamera, int rm, iggFrame *parent, int index, const iObjectKey *indkey) : iggWidgetKeyHandlerBase(WidgetType::Entry,dirkey,rm,parent,index,indkey), iggRenderWindowObserver(parent,followCamera)
{
	if(this->GetKey().Argument()!=iObjectKey::_Float || this->GetKey().Dimension()<2 || this->GetKey().Dimension()>3)
	{
		IERROR_LOW("Widget iggWidgetKeyTrackBall is congifured incorrectly.");
	}

	mIsVectorKey = (this->GetKey().Dimension() == 3);
	mSubject = iggSubjectFactory::CreateWidgetTrackBallSubject(this,followCamera);

	if(rm != RenderMode::NoRender)
	{
		iggWidgetRenderModeButton *rm = new iggWidgetRenderModeButton(this); IERROR_ASSERT(rm);
		rm->ClearLaidOutFlag(mSubject);
		mSubject->AddButton(rm->GetSubject());
	}

	this->SetRenderMode(rm);
}


iggWidgetKeyTrackBall::~iggWidgetKeyTrackBall()
{
}


void iggWidgetKeyTrackBall::UpdateWidgetBody()
{
	this->UpdateHelp();

	if(mIsVectorKey)
	{
		float n[3];
		if(this->GetShell()->GetControlModule()->QueryValue(this->GetKey(),n,3))
		{
			this->SetDirection(n);
		}
	}
	else
	{
		float a[2];
		if(this->GetShell()->GetControlModule()->QueryValue(this->GetKey(),a,2))
		{
			this->SetAngles(a[0],a[1]);
		}
	}
}


void iggWidgetKeyTrackBall::OnRenderWindowModified()
{
	mSubject->Repaint();
}


void iggWidgetKeyTrackBall::OnBool1Body(bool final)
{
	iString ws;
	float dir[3];

	mSubject->GetDirection(dir);

	if(mIsVectorKey)
	{
		this->GetShell()->GetControlModule()->PackCommand(ws,this->GetKey(),dir,3);
	}
	else
	{
		float a[2];
		//
		//  Use vtkLight angle convention
		//
#ifdef IVTK_PRE54
		a[0] = vtkMath::RadiansToDegrees()*atan2(dir[1],sqrt(dir[0]*dir[0]+dir[2]*dir[2])); 
		a[1] = vtkMath::RadiansToDegrees()*atan2(dir[0],dir[2]);
#else
		a[0] = vtkMath::DegreesFromRadians(atan2(dir[1],sqrt(dir[0]*dir[0]+dir[2]*dir[2]))); 
		a[1] = vtkMath::DegreesFromRadians(atan2(dir[0],dir[2]));
#endif
		this->GetShell()->GetControlModule()->PackCommand(ws,this->GetKey(),a,2);
	}

	bool ret = this->GetShell()->GetControlModule()->Execute(ws,(final || this->GetRenderMode()==RenderMode::Immediate) && this->GetMainWindow()->IsAutoRender(),this->GetExecuteFlags());
	if(!ret) IERROR_LOW("iggWidgetKeyTrackBall::OnDirection failed.");
}


void iggWidgetKeyTrackBall::SetDirection(float x, float y, float z)
{
	mSubject->SetDirection(x,y,z);
}


void iggWidgetKeyTrackBall::SetAngles(float elevation, float azimuth)
{
	//
	//  Use vtkLight angle convention
	//
#ifdef IVTK_PRE54
	elevation *= vtkMath::DegreesToRadians();
	azimuth *= vtkMath::DegreesToRadians();
#else
	elevation = vtkMath::RadiansFromDegrees(elevation);
	azimuth = vtkMath::RadiansFromDegrees(azimuth);
#endif
	mSubject->SetDirection(cos(elevation)*sin(azimuth),sin(elevation),cos(elevation)*cos(azimuth));
}

#endif
