/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     lagrange_2_2d.c                                                */
/*                                                                          */
/* description:  piecewise quadratic Lagrange elements in 2d                */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#define N_BAS2_2D  6    /* (N_VERTICES_2D + N_EDGES_2D) */

static const REAL   bary2_2d[N_BAS2_2D][N_LAMBDA] = {{1.0, 0.0, 0.0, 0.0},
						     {0.0, 1.0, 0.0, 0.0},
						     {0.0, 0.0, 1.0, 0.0},
						     {0.0, 0.5, 0.5, 0.0},
						     {0.5, 0.0, 0.5, 0.0},
						     {0.5, 0.5, 0.0, 0.0}};

/****************************************************************************/
/*  basisfunction at vertex 0                                               */
/****************************************************************************/

static REAL phi2v0_2d(const REAL lambda[N_LAMBDA])
{
  return(lambda[0]*(2.0*lambda[0] - 1.0));
}

static const REAL *grd_phi2v0_2d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};

  grd[0] = 4.0*lambda[0] - 1.0;
  return((const REAL *) grd);
}

static const REAL (*D2_phi2v0_2d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{4, 0, 0, 0},
					      {0, 0, 0, 0},
					      {0, 0, 0, 0},
					      {0, 0, 0, 0}};

  return(D2);
}

/****************************************************************************/
/*  basisfunction at vertex 1                                               */
/****************************************************************************/

static REAL phi2v1_2d(const REAL lambda[N_LAMBDA])
{
  return(lambda[1]*(2.0*lambda[1] - 1.0));
}

static const REAL *grd_phi2v1_2d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};

  grd[1] = 4.0*lambda[1] - 1.0;
  return((const REAL *) grd);
}

static const REAL (*D2_phi2v1_2d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 0, 0, 0},
					      {0, 4, 0, 0},
					      {0, 0, 0, 0},
					      {0, 0, 0, 0}};

  return(D2);
}

/****************************************************************************/
/*  basisfunction at vertex 2                                               */
/****************************************************************************/

static REAL phi2v2_2d(const REAL lambda[N_LAMBDA])
{
  return(lambda[2]*(2.0*lambda[2] - 1.0));
}

static const REAL *grd_phi2v2_2d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};

  grd[2] = 4.0*lambda[2] - 1.0;
  return((const REAL *) grd);
}

static const REAL (*D2_phi2v2_2d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 0, 0, 0},
					      {0, 0, 0, 0},
					      {0, 0, 4, 0},
					      {0, 0, 0, 0}};

  return(D2);
}

/****************************************************************************/
/*  basisfunction at edge 0                                                 */
/****************************************************************************/

static REAL phi2e0_2d(const REAL lambda[N_LAMBDA])
{
  return(4.0*lambda[1]*lambda[2]);
}

static const REAL *grd_phi2e0_2d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};

  grd[1] = 4.0*lambda[2];
  grd[2] = 4.0*lambda[1];
  return((const REAL *) grd);
}

static const REAL (*D2_phi2e0_2d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 0, 0, 0},
					      {0, 0, 4, 0},
					      {0, 4, 0, 0},
					      {0, 0, 0, 0}};
  return(D2);
}

/****************************************************************************/
/*  basisfunction at edge 1                                                 */
/****************************************************************************/

static REAL phi2e1_2d(const REAL lambda[N_LAMBDA])
{
  return(4.0*lambda[0]*lambda[2]);
}

static const REAL *grd_phi2e1_2d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};

  grd[0] = 4.0*lambda[2];
  grd[2] = 4.0*lambda[0];
  return((const REAL *) grd);
}

static const REAL (*D2_phi2e1_2d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 0, 4, 0},
					      {0, 0, 0, 0},
					      {4, 0, 0, 0},
					      {0, 0, 0, 0}};
  return(D2);
}

/****************************************************************************/
/*  basisfunction at edge 2                                                 */
/****************************************************************************/

static REAL phi2e2_2d(const REAL lambda[N_LAMBDA])
{
  return(4.0*lambda[0]*lambda[1]);
}

static const REAL *grd_phi2e2_2d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};

  grd[0] = 4.0*lambda[1];
  grd[1] = 4.0*lambda[0];
  return((const REAL *) grd);
}

static const REAL (*D2_phi2e2_2d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 4, 0, 0},
					      {4, 0, 0, 0},
					      {0, 0, 0, 0},
					      {0, 0, 0, 0}};
  return(D2);
}

/****************************************************************************/
/*  functions for combining basisfunctions with coefficients                */
/****************************************************************************/

static const DOF *get_dof_indices2_2d(const EL *el, const DOF_ADMIN *admin,
				   DOF *idof)
{
  static DOF  index_vec[N_BAS2_2D];
  int         i, n0;
  DOF         *rvec = idof ? idof : index_vec;
  DOF         **dof = el->dof;

  n0 = admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = dof[i][n0];
  n0 = admin->n0_dof[EDGE];
  for (i = N_VERTICES_2D; i < N_BAS2_2D; i++)
    rvec[i] = dof[i][n0];

  return((const DOF *) rvec);
}

static const S_CHAR *get_bound2_2d(const EL_INFO *el_info, S_CHAR *bound)
{
  FUNCNAME("get_bound2_2d");
  static S_CHAR  bound_vec[N_BAS2_2D];
  S_CHAR         *rvec = bound ? bound : bound_vec;
  int            i;

  TEST_FLAG(FILL_BOUND, el_info);

  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = el_info->vertex_bound[i];
  for (i = 0; i < N_EDGES_2D; i++)
    rvec[N_VERTICES_2D+i] = el_info->edge_bound[i];

  return((const S_CHAR *) rvec);
}

static const int *get_int_vec2_2d(const EL *el, const DOF_INT_VEC *vec,
				int *ivec)
{
  FUNCNAME("get_int_vec2_2d");
  int           i, n0;
  static int    local_vec[6];
  int           *v = nil, *rvec = ivec ? ivec : local_vec;
  DOF           **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = v[dof[i][n0]];
  n0 = vec->fe_space->admin->n0_dof[EDGE];
  for (i = N_VERTICES_2D; i < N_BAS2_2D; i++)
    rvec[i] = v[dof[i][n0]];

  return((const int *) rvec);
}

static const REAL *get_real_vec2_2d(const EL *el, const DOF_REAL_VEC *vec,
				    REAL *Rvec)
{
  FUNCNAME("get_real_vec2_2d");
  int            i, n0;
  static REAL    local_vec[6];
  REAL           *v = nil, *rvec = Rvec ? Rvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = v[dof[i][n0]];
  n0 = vec->fe_space->admin->n0_dof[EDGE];
  for (i = N_VERTICES_2D; i < N_BAS2_2D; i++)
    rvec[i] = v[dof[i][n0]];

  return((const REAL *) rvec);
}

static const REAL_D *get_real_d_vec2_2d(const EL *el, 
					const DOF_REAL_D_VEC *vec,
					REAL_D *Rvec)
{
  FUNCNAME("get_real_d_vec2_2d");
  int            i, k, n0;
  static REAL_D  local_vec[6];
  REAL_D         *v = nil, *rvec = Rvec ? Rvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_2D; i++)
    for (k = 0; k < DIM_OF_WORLD; k++)
      rvec[i][k] = v[dof[i][n0]][k];
  n0 = vec->fe_space->admin->n0_dof[EDGE];
  for (i = N_VERTICES_2D; i < N_BAS2_2D; i++)
    for (k = 0; k < DIM_OF_WORLD; k++)
      rvec[i][k] = v[dof[i][n0]][k];

  return((const REAL_D *) rvec);
}

static const U_CHAR *get_uchar_vec2_2d(const EL *el, const DOF_UCHAR_VEC *vec,
				       U_CHAR *uvec)
{
  FUNCNAME("get_uchar_vec2_2d");
  int            i, n0;
  static U_CHAR  local_vec[6];
  U_CHAR         *v = nil, *rvec = uvec ? uvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = v[dof[i][n0]];
  n0 = vec->fe_space->admin->n0_dof[EDGE];
  for (i = N_VERTICES_2D; i < N_BAS2_2D; i++)
    rvec[i] = v[dof[i][n0]];

  return((const U_CHAR *) rvec);
}

static const S_CHAR *get_schar_vec2_2d(const EL *el, const DOF_SCHAR_VEC *vec,
				       S_CHAR *svec)
{
  FUNCNAME("get_schar_vec2_2d");
  int            i, n0;
  static S_CHAR  local_vec[6];
  S_CHAR         *v = nil, *rvec = svec ? svec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = v[dof[i][n0]];
  n0 = vec->fe_space->admin->n0_dof[EDGE];
  for (i = N_VERTICES_2D; i < N_BAS2_2D; i++)
    rvec[i] = v[dof[i][n0]];

  return((const S_CHAR *) rvec);
}


/*--------------------------------------------------------------------*/
/*--- function for local interpolaton of scalar functions          ---*/
/*--------------------------------------------------------------------*/

GENERATE_INTERPOL(,2,2,6)

/*--------------------------------------------------------------------*/
/*--- function for local interpolaton of vector functions          ---*/
/*--------------------------------------------------------------------*/

GENERATE_INTERPOL_D(,2,2,6)


/****************************************************************************/
/*  functions for interpolation/ restriction during refinement/coarsening   */
/****************************************************************************/

static void real_refine_inter2_2d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("real_refine_inter2_2d");
  EL        *el;
  REAL      *vec = nil;
  int       node, n0;
  DOF       cdof;
  const DOF *pdof;
  const DOF       *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;

  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(vec, drv);
  if (!drv->fe_space)
  {
    ERROR("no fe_space in dof_real_vec %s\n", NAME(drv));
    return;
  }
  else if (!drv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drv->fe_space));
    return;
  }
  get_dof_indices = drv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drv->fe_space);

  pdof = get_dof_indices(el, admin, nil);
  
  node = drv->fe_space->mesh->node[VERTEX];        
  n0 = admin->n0_dof[VERTEX];

/****************************************************************************/
/*  newest vertex of child[0] and child[1]                                  */
/****************************************************************************/

  cdof = el->child[0]->dof[node+2][n0];  /*      newest vertex is 2 */
  vec[cdof] = vec[pdof[5]];

  node = drv->fe_space->mesh->node[EDGE];        
  n0 = admin->n0_dof[EDGE];

/****************************************************************************/
/*  midpoint of edge on child[0] at the refinement edge                     */
/****************************************************************************/
  
  cdof = el->child[0]->dof[node][n0]; 
  vec[cdof] = 0.375*vec[pdof[0]] - 0.125*vec[pdof[1]]  + 0.75*vec[pdof[5]];

/****************************************************************************/
/* node in the common edge of child[0] and child[1]                         */
/****************************************************************************/

  cdof = el->child[0]->dof[node+1][n0]; 
  vec[cdof] = -0.125*(vec[pdof[0]] + vec[pdof[1]]) + 0.25*vec[pdof[5]]
    + 0.5*(vec[pdof[3]] + vec[pdof[4]]);

/****************************************************************************/
/*  midpoint of edge on child[1] at the refinement edge                     */
/****************************************************************************/
  
  cdof = el->child[1]->dof[node+1][n0]; 
  vec[cdof] = -0.125*vec[pdof[0]] + 0.375*vec[pdof[1]]  + 0.75*vec[pdof[5]];

  if (n > 1)
  {
/****************************************************************************/
/*  adjust the value at the midpoint of the common edge of neigh's children */
/****************************************************************************/
    el = list[1].el_info.el;
    pdof = get_dof_indices(el, admin, nil);

    cdof = el->child[0]->dof[node+1][n0]; 
    vec[cdof] = -0.125*(vec[pdof[0]] + vec[pdof[1]]) + 0.25*vec[pdof[5]]
      + 0.5*(vec[pdof[3]] + vec[pdof[4]]);
  }
  return;
}

static void real_coarse_inter2_2d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("real_coarse_inter2_2d");
  EL        *el;
  REAL      *v = nil;
  DOF       cdof, pdof;
  const DOF_ADMIN *admin;
  MESH      *mesh = nil;

  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(v, drv);
  if (!drv->fe_space)
  {
    ERROR("no fe_space in dof_real_vec %s\n", NAME(drv));
    return;
  }
  else if (!drv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drv->fe_space));
    return;
  }
  GET_STRUCT(admin,drv->fe_space);
  GET_STRUCT(mesh,drv->fe_space);

/****************************************************************************/
/*  values on child[0]                                                      */
/****************************************************************************/

  cdof = el->child[0]->dof[mesh->node[VERTEX]+2][admin->n0_dof[VERTEX]]; 
  pdof = el->dof[mesh->node[EDGE]+2][admin->n0_dof[EDGE]]; 
  v[pdof] = v[cdof];

  return;
}

static void real_coarse_restr2_2d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("real_coarse_restr2_2d");
  EL        *el;
  REAL      *vec = nil;
  int       node, n0;
  DOF       cdof2, cdof3, cdof4;
  const DOF *pdof;
  const DOF       *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;

  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(vec, drv);
  if (!drv->fe_space)
  {
    ERROR("no fe_space in dof_real_vec %s\n", NAME(drv));
    return;
  }
  else if (!drv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drv->fe_space));
    return;
  }
  get_dof_indices = drv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drv->fe_space);

  pdof = get_dof_indices(el, admin, nil);

/****************************************************************************/
/*  contributions of dofs located on child[0]                               */
/****************************************************************************/

  node = drv->fe_space->mesh->node[VERTEX];        
  n0 = admin->n0_dof[VERTEX];
  cdof2 = el->child[0]->dof[node+2][n0];

  node = drv->fe_space->mesh->node[EDGE];        
  n0 = admin->n0_dof[EDGE];
  cdof3 = el->child[0]->dof[node][n0];
  cdof4 = el->child[0]->dof[node+1][n0];

  vec[pdof[0]] += 0.375*vec[cdof3] - 0.125*vec[cdof4];
  vec[pdof[1]] += -0.125*(vec[cdof3] + vec[cdof4]);
  vec[pdof[3]] += 0.5*vec[cdof4];
  vec[pdof[4]] += 0.5*vec[cdof4];
  vec[pdof[5]] = vec[cdof2] + 0.75*vec[cdof3] + 0.25*vec[cdof4];

/****************************************************************************/
/*  contributions of dofs located on child[1] and not on child[0]           */
/****************************************************************************/

  cdof4 = el->child[1]->dof[node+1][n0];

  vec[pdof[0]] += -0.125*vec[cdof4];
  vec[pdof[1]] += 0.375*vec[cdof4];
  vec[pdof[5]] += 0.75*vec[cdof4];

  if (n > 1)
  {
    el = list[1].el_info.el;
    pdof = get_dof_indices(el, admin, nil);
    
/****************************************************************************/
/*  first set those values not effected by previous element                 */
/****************************************************************************/

    cdof4 = el->child[0]->dof[node+1][n0];
    vec[pdof[3]] += 0.5*vec[cdof4];
    vec[pdof[4]] += 0.5*vec[cdof4];

/****************************************************************************/
/*  and now, update the values in the refinement edge                       */
/****************************************************************************/

    vec[pdof[0]] += -0.125*vec[cdof4];
    vec[pdof[1]] += -0.125*vec[cdof4];
    vec[pdof[5]] += 0.25*vec[cdof4];
  }
  return;
}

static void real_d_refine_inter2_2d(DOF_REAL_D_VEC *drdv, RC_LIST_EL *list,
				    int n)
{
  FUNCNAME("real_d_refine_inter2_2d");
  EL        *el;
  REAL_D    *vec = nil;
  int       node, n0, j;
  DOF       cdof;
  const DOF *pdof;
  const DOF       *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;

  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(vec, drdv);
  if (!drdv->fe_space)
  {
    ERROR("no fe_space in dof_real_d_vec %s\n", NAME(drdv));
    return;
  }
  else if (!drdv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drdv->fe_space));
    return;
  }
  get_dof_indices = drdv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drdv->fe_space);

  pdof = get_dof_indices(el, admin, nil);
  
  node = drdv->fe_space->mesh->node[VERTEX];        
  n0 = admin->n0_dof[VERTEX];

/****************************************************************************/
/*  newest vertex of child[0] and child[1]                                  */
/****************************************************************************/

  cdof = el->child[0]->dof[node+2][n0];  /*      newest vertex is 2 */
  for (j = 0; j < DIM_OF_WORLD; j++)
    vec[cdof][j] = vec[pdof[5]][j];

  node = drdv->fe_space->mesh->node[EDGE];        
  n0 = admin->n0_dof[EDGE];

/****************************************************************************/
/*  midpoint of edge on child[0] at the refinement edge                     */
/****************************************************************************/
  
  cdof = el->child[0]->dof[node][n0]; 
  for (j = 0; j < DIM_OF_WORLD; j++)
    vec[cdof][j] = (0.375*vec[pdof[0]][j] - 0.125*vec[pdof[1]][j]
		    + 0.75*vec[pdof[5]][j]);

/****************************************************************************/
/* node in the common edge of child[0] and child[1]                         */
/****************************************************************************/

  cdof = el->child[0]->dof[node+1][n0]; 
  for (j = 0; j < DIM_OF_WORLD; j++)
    vec[cdof][j] = -0.125*(vec[pdof[0]][j] + vec[pdof[1]][j]) 
      + 0.25*vec[pdof[5]][j] + 0.5*(vec[pdof[3]][j] + vec[pdof[4]][j]);

/****************************************************************************/
/*  midpoint of edge on child[1] at the refinement edge                     */
/****************************************************************************/
  
  cdof = el->child[1]->dof[node+1][n0]; 
  for (j = 0; j < DIM_OF_WORLD; j++)
    vec[cdof][j] = (-0.125*vec[pdof[0]][j] + 0.375*vec[pdof[1]][j]
		    + 0.75*vec[pdof[5]][j]);

  if (n > 1)
  {
/****************************************************************************/
/*  adjust the value at the midpoint of the common edge of neigh's children */
/****************************************************************************/
    el = list[1].el_info.el;
    pdof = get_dof_indices(el, admin, nil);

    cdof = el->child[0]->dof[node+1][n0]; 
    for (j = 0; j < DIM_OF_WORLD; j++)
      vec[cdof][j] = -0.125*(vec[pdof[0]][j] + vec[pdof[1]][j])
	+ 0.25*vec[pdof[5]][j]+ 0.5*(vec[pdof[3]][j] + vec[pdof[4]][j]);
  }
  return;
}

static void real_d_coarse_inter2_2d(DOF_REAL_D_VEC *drdv, RC_LIST_EL *list,
				    int n)
{
  FUNCNAME("real_d_coarse_inter2_2d");
  EL        *el;
  REAL_D    *v = nil;
  int       j;
  DOF       cdof, pdof;
  MESH      *mesh = nil;
  const DOF_ADMIN *admin;

  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(v, drdv);
  if (!drdv->fe_space)
  {
    ERROR("no fe_space in dof_real_d_vec %s\n", NAME(drdv));
    return;
  }
  else if (!drdv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drdv->fe_space));
    return;
  }
  GET_STRUCT(admin,drdv->fe_space);
  GET_STRUCT(mesh,drdv->fe_space);

/****************************************************************************/
/*  values on child[0]                                                      */
/****************************************************************************/

  cdof = el->child[0]->dof[mesh->node[VERTEX]+2][admin->n0_dof[VERTEX]]; 
  pdof = el->dof[mesh->node[EDGE]+2][admin->n0_dof[EDGE]]; 
  for (j = 0; j < DIM_OF_WORLD; j++)
    v[pdof][j] = v[cdof][j];

  return;
}

static void real_d_coarse_restr2_2d(DOF_REAL_D_VEC *drdv, RC_LIST_EL *list,
				    int n)
{
  FUNCNAME("real_d_coarse_restr2_2d");
  EL        *el;
  REAL_D    *vec = nil;
  int       node, n0, j;
  DOF       cdof2, cdof3, cdof4;
  const DOF *pdof;
  const DOF       *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;

  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(vec, drdv);
  if (!drdv->fe_space)
  {
    ERROR("no fe_space in dof_real_d_vec %s\n", NAME(drdv));
    return;
  }
  else if (!drdv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drdv->fe_space));
    return;
  }
  get_dof_indices = drdv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drdv->fe_space);

  pdof = get_dof_indices(el, admin, nil);

/****************************************************************************/
/*  contributions of dofs located on child[0]                               */
/****************************************************************************/

  node = drdv->fe_space->mesh->node[VERTEX];        
  n0 = admin->n0_dof[VERTEX];
  cdof2 = el->child[0]->dof[node+2][n0];

  node = drdv->fe_space->mesh->node[EDGE];        
  n0 = admin->n0_dof[EDGE];
  cdof3 = el->child[0]->dof[node][n0];
  cdof4 = el->child[0]->dof[node+1][n0];

  for (j = 0; j < DIM_OF_WORLD; j++)
  {  
    vec[pdof[0]][j] += 0.375*vec[cdof3][j] - 0.125*vec[cdof4][j];
    vec[pdof[1]][j] += -0.125*(vec[cdof3][j] + vec[cdof4][j]);
    vec[pdof[3]][j] += 0.5*vec[cdof4][j];
    vec[pdof[4]][j] += 0.5*vec[cdof4][j];
    vec[pdof[5]][j] = vec[cdof2][j] + 0.75*vec[cdof3][j] + 0.25*vec[cdof4][j];
  }
/****************************************************************************/
/*  contributions of dofs located on child[1] and not on child[0]           */
/****************************************************************************/

  cdof4 = el->child[1]->dof[node+1][n0];

  for (j = 0; j < DIM_OF_WORLD; j++)
  {  
    vec[pdof[0]][j] += -0.125*vec[cdof4][j];
    vec[pdof[1]][j] += 0.375*vec[cdof4][j];
    vec[pdof[5]][j] += 0.75*vec[cdof4][j];
  }
  if (n > 1)
  {
    el = list[1].el_info.el;
    pdof = get_dof_indices(el, admin, nil);
    
/****************************************************************************/
/*  first set those values not effected by previous element                 */
/****************************************************************************/

    cdof4 = el->child[0]->dof[node+1][n0];

    for (j = 0; j < DIM_OF_WORLD; j++)
    {  
      vec[pdof[3]][j] += 0.5*vec[cdof4][j];
      vec[pdof[4]][j] += 0.5*vec[cdof4][j];

/****************************************************************************/
/*  and now, update the values in the refinement edge                       */
/****************************************************************************/

      vec[pdof[0]][j] += -0.125*vec[cdof4][j];
      vec[pdof[1]][j] += -0.125*vec[cdof4][j];
      vec[pdof[5]][j] += 0.25*vec[cdof4][j];
    }
  }
  return;
}

static BAS_FCT      *phi2_2d[6]     = {phi2v0_2d, phi2v1_2d, phi2v2_2d,
				       phi2e0_2d, phi2e1_2d, phi2e2_2d};
static GRD_BAS_FCT  *grd_phi2_2d[6] = {grd_phi2v0_2d, grd_phi2v1_2d,
				       grd_phi2v2_2d, grd_phi2e0_2d,
				       grd_phi2e1_2d, grd_phi2e2_2d};
static D2_BAS_FCT   *D2_phi2_2d[6]  = {D2_phi2v0_2d, D2_phi2v1_2d,
				       D2_phi2v2_2d, D2_phi2e0_2d,
				       D2_phi2e1_2d, D2_phi2e2_2d};

static BAS_FCTS  lagrange2_2d = {"lagrange2_2d", 2, 6, 2,
				 {1, 0, 1, 0}, /* VERTEX, CENTER, EDGE, FACE */
				 nil,
				 phi2_2d, grd_phi2_2d, D2_phi2_2d, 
				 get_dof_indices2_2d, 
				 get_bound2_2d,
				 interpol2_2d,
				 interpol_d2_2d,
				 get_int_vec2_2d,
				 get_real_vec2_2d,
				 get_real_d_vec2_2d,
				 get_uchar_vec2_2d,
				 get_schar_vec2_2d,
				 real_refine_inter2_2d,
				 real_coarse_inter2_2d,
				 real_coarse_restr2_2d,
				 real_d_refine_inter2_2d,
				 real_d_coarse_inter2_2d,
				 real_d_coarse_restr2_2d,
				 bary2_2d,};
