/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     oem_solve_d.c                                                  */
/*                                                                          */
/* description:  matrix-vector multiplication, standard preconditioning     */
/*               and standard solver interface to OEM-library for decoupled */
/*               vector valued problems:                                    */
/*               A u_n = f_n, n = 1,...,DIM_OF_WORLD, where                 */
/*               A is stored in a DOF_MATRIX, u and f in DOF_REAL_D_VECs    */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#include "alberta.h"

struct mv_data
{
  const DOF_MATRIX    *matrix;
  MatrixTranspose     transpose;

  const DOF_SCHAR_VEC *bound;

  struct mv_data      *next;
};

/*---8<---------------------------------------------------------------------*/
/*---   y <- A x  or y <- A^T x                                          ---*/
/*--------------------------------------------------------------------->8---*/

int mat_vec_d(void *ud, int dim, const REAL *x, REAL *y)
{
  FUNCNAME("mat_vec_d");
  DOF_REAL_D_VEC  dof_x = {nil, nil, "x", 0, nil, nil, nil};
  DOF_REAL_D_VEC  dof_y = {nil, nil, "y", 0, nil, nil, nil};
  struct mv_data  *data     = (struct mv_data *)ud;
  const S_CHAR    *b        = data->bound ? data->bound->vec : nil;
  const DOF_ADMIN *admin    = data->matrix->row_fe_space->admin;
  REAL_D          *xd = (REAL_D *) x, *yd = (REAL_D *) y;

  TEST_EXIT(dim == DIM_OF_WORLD*admin->size_used,
"argument dim != DIM_OF_WORLD*data->matrix->row_fe_space->admin->size_used\n");

  dof_x.fe_space = data->matrix->col_fe_space;
  dof_y.fe_space = data->matrix->row_fe_space;
  dof_x.size = dof_y.size = admin->size_used;
  dof_x.vec = xd;
  dof_y.vec = yd;

  dof_mv_d(data->transpose, data->matrix, &dof_x, &dof_y);

  if (b)
  {
    FOR_ALL_DOFS(admin, if (b[dof] > 0)  COPY_DOW(xd[dof], yd[dof]));
  }
  return(0);
}

void *init_mat_vec_d(MatrixTranspose transpose, const DOF_MATRIX *A, 
		     const DOF_SCHAR_VEC *bound)
{
  FUNCNAME("init_mat_vec_d");
  static struct mv_data *first = nil;
  struct mv_data        *data;

  for (data = first; data; data = data->next)
    if (data->matrix == A  &&  data->bound == bound  && data->transpose == transpose)
      break;

  if ((transpose == Transpose) && bound)
    ERROR_EXIT("don't know how to handle bound vector for y = A^T x\n");

  if (!data) 
  {
    data = MEM_ALLOC(1, struct mv_data);

    data->matrix    = A;
    data->transpose = transpose;

    data->bound     = bound;

    data->next      = first;
    first           = data;
  }

  return(data);
}

void exit_mat_vec_d(void *data)
{
  return;
}

/*---8<---------------------------------------------------------------------*/
/*---   diaoganal preconditioning for decoupled vector valued  problems  ---*/
/*--------------------------------------------------------------------->8---*/

struct precon_data
{
  PRECON              precon;

  const DOF_MATRIX    *matrix;
  const DOF_SCHAR_VEC *bound;
  int                 dim;

  const REAL          *diag_1;
  size_t              size_diag_1;

  struct precon_data  *next;
};

static void diag_precon_d(void *ud, int dim, REAL *r)
{
  FUNCNAME("diag_precon_d");
  struct precon_data  *data = (struct precon_data *)ud;
  int                 i, n;
  const REAL          *diag_1 = data->diag_1;
  REAL_D              *rd = (REAL_D *) r;

  DEBUG_TEST_EXIT(dim == data->dim*DIM_OF_WORLD,
		  "argument dim != precon_data->dim*DIM_OF_WORLD\n");

  for (i = 0; i < data->dim; i++)
  {
    for (n = 0; n < DIM_OF_WORLD; n++)
      rd[i][n] *= diag_1[i];
  }

  return;
}

/*--------------------------------------------------------------------------*/
/*---  initialize diagonal preconditioning  --------------------------------*/
/*--------------------------------------------------------------------------*/

static int init_diag_precon_d(void *precon_data)
{
  FUNCNAME("init_diag_precon_d");
  struct precon_data        *data  = (struct precon_data *)precon_data;
  MATRIX_ROW                **row  = data->matrix->matrix_row;
  const S_CHAR              *b     = data->bound ? data->bound->vec : nil;
  const DOF_ADMIN           *admin = data->matrix->row_fe_space->admin;
  int                       dim    = admin->size_used;
  REAL                      *diag_1;

  if (data->size_diag_1 < (size_t) dim)
  {
    data->diag_1 = MEM_REALLOC(data->diag_1, data->size_diag_1, dim, REAL);
    data->size_diag_1 = dim;
  }

  data->dim = dim;
  diag_1 = (REAL *) data->diag_1;

  if (b)
  {
    FOR_ALL_DOFS(admin, 
 diag_1[dof] = (row[dof] && b[dof] <= 0 && ABS(row[dof]->entry[0]) > 1.0E-20) ? 1.0/row[dof]->entry[0] : 1.0);
  }
  else
  {
    FOR_ALL_DOFS(admin, diag_1[dof] = (row[dof] && ABS(row[dof]->entry[0]) > 1.0E-20) ? 1.0/row[dof]->entry[0] : 1.0);
  }

  FOR_ALL_FREE_DOFS(admin,
		    if (dof >=  admin->size_used){
		      break;
		    }
		    diag_1[dof] = 0.0);

  return true;
}

static void exit_diag_precon_d(void *precon_data)
{
  struct precon_data *data = (struct precon_data *)precon_data;

  if (data->diag_1)
  {
    MEM_FREE(data->diag_1, data->size_diag_1, REAL);
    data->diag_1 = nil;
    data->size_diag_1 = 0;
  }
  data->dim = 0;

  return;
}

const PRECON *get_diag_precon_d(const DOF_MATRIX *A, const DOF_SCHAR_VEC *bound)
{
  static struct precon_data *first = nil;
  struct precon_data        *data;

  for (data = first; data; data = data->next)
    if (data->matrix == A  &&  data->bound == bound)  break;

  if (!data)
  {
    data = MEM_CALLOC(1, struct precon_data);

    TEST_EXIT(A->row_fe_space == A->col_fe_space,
	      "Row and column FE_SPACEs don't match!\n");

    data->precon.precon_data = data;
    data->precon.init_precon = init_diag_precon_d;
    data->precon.precon      = diag_precon_d;
    data->precon.exit_precon = exit_diag_precon_d;

    data->matrix   = A;
    data->bound    = bound;

    data->next     = first;
    first          = data;
  }

  return(&data->precon);
}

/*--------------------------------------------------------------------------*/
/*---  interface to oem solver for decoupled vector valued problems  -------*/
/*--------------------------------------------------------------------------*/

int oem_solve_d(const DOF_MATRIX *A, const DOF_REAL_D_VEC *f, 
		DOF_REAL_D_VEC *u, OEM_SOLVER solver, REAL tol,
		int icon, int restart, int max_iter, int info)
{
  FUNCNAME("oem_solve_d");
  int          n, iter = -1, dim, size_used;
  size_t       size;
  OEM_DATA     oem = {nil};
  const PRECON *precon = nil;
  WORKSPACE    ws = {0, nil};

  TEST_EXIT(A->row_fe_space == A->col_fe_space,
	    "Row and column FE_SPACEs don't match!\n");

  size_used = A->row_fe_space->admin->size_used;
  FOR_ALL_FREE_DOFS(A->row_fe_space->admin, 
		    if (dof < size_used)
		    {
		      for (n = 0; n < DIM_OF_WORLD; n++)
			u->vec[dof][n] = f->vec[dof][n] = 0.0;
		    });

  oem.mat_vec = mat_vec_d;
  oem.mat_vec_data = init_mat_vec_d(NoTranspose, A, nil);

  switch(icon)
  {
  case 0:
    oem.left_precon = nil;
    break;
  case 2:
    precon = get_HB_precon_d(A->row_fe_space, nil, 1, info);
    break;
  case 3:
    precon = get_BPX_precon_d(A->row_fe_space, nil, 1, info);
    break;
  default:
    precon = get_diag_precon_d(A, nil);
    break;
  }

  if (precon)
  {
    if (precon->init_precon && !(*precon->init_precon)(precon->precon_data)) {
#if 0
      /* init_precon() has already cleaned up after itself */
      if (precon->exit_precon) {
	(*precon->exit_precon)(precon->precon_data);
      }
#endif
      precon = nil;
      MSG("init_precon() failed, disabling preconditioner!\n");
    } else {
      oem.left_precon_data = precon->precon_data;
      oem.left_precon = precon->precon;
    }
  }

  oem.ws = &ws;
  oem.tolerance = tol;
  oem.restart   = 0;
  oem.max_iter  = max_iter;
  oem.info      = info;

  dim = size_used*DIM_OF_WORLD;
  switch (solver)
  {
  case BiCGStab:
    REALLOC_WORKSPACE(&ws, 5*dim*sizeof(REAL));
    iter = oem_bicgstab(&oem, dim, (REAL *) f->vec, (REAL *) u->vec);
    break;
  case CG:
    REALLOC_WORKSPACE(&ws, 3*dim*sizeof(REAL));
    iter = oem_cg(&oem, dim, (REAL *) f->vec, (REAL *) u->vec);
    break;
  case TfQMR:
    REALLOC_WORKSPACE(&ws, 11*dim*sizeof(REAL));
    iter = oem_tfqmr(&oem, dim, (REAL *) f->vec, (REAL *) u->vec);
    break;
  case GMRes:
    oem.restart = restart = MAX(0, MIN(restart, dim));
    size = ((restart+2)*dim + restart*(restart+4))*sizeof(REAL);
    REALLOC_WORKSPACE(&ws, size);
    iter = oem_gmres(&oem, dim, (REAL *) f->vec, (REAL *) u->vec);
    break;
  case ODir:
    REALLOC_WORKSPACE(&ws, 6*dim*sizeof(REAL));
    iter = oem_odir(&oem, dim, (REAL *) f->vec, (REAL *) u->vec);
    break;
  case ORes:
    REALLOC_WORKSPACE(&ws, 7*dim*sizeof(REAL));
    iter = oem_ores(&oem, dim, (REAL *) f->vec, (REAL *) u->vec);
    break;
  default:
    ERROR_EXIT("unknown OEM solver %d\n", (int) solver);
  }

  CLEAR_WORKSPACE(&ws);

  if (precon && precon->exit_precon)
    (*precon->exit_precon)(precon->precon_data);

  return(iter);
}
